<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.1/2745343
 */

namespace Authentication\Helper;

use Application\Cache\AbstractCacheService;
use Application\Config\ConfigException;
use Application\Config\ConfigManager;
use Application\Config\IConfigDefinition;
use Application\Config\IDao;
use Application\Config\Services;
use Application\Connection\ConnectionFactory;
use Application\Factory\InvokableService;
use Application\Log\SwarmLogger;
use Application\Model\IModelDAO;
use Authentication\Service\Auth;
use Interop\Container\ContainerInterface;
use InvalidArgumentException;
use Laminas\Authentication\Result;
use P4\Connection\ConnectionInterface;
use P4\Connection\Exception\ServiceNotFoundException;
use Authentication\Adapter;
use Users\Filter\User as UserFilter;

class LoginHelper implements InvokableService, ILoginHelper
{
    const LOG_PREFIX = LoginHelper::class;
    private $services;

    public function __construct(ContainerInterface $services, array $options = null)
    {
        $this->services = $services;
    }

    /**
     * @inheritDoc
     */
    public function invalidateCache(string $authUser, ConnectionInterface $p4Admin)
    {
        $userDao = $this->services->get(IModelDAO::USER_DAO);
        // Get authenticated user object and invalidate user cache if
        // authenticated user is not in cache - this most likely means
        // that user has been added to Perforce but the form-commit
        // user trigger was not fired
        if (!$userDao->exists($authUser, $p4Admin)) {
            try {
                $userDao->fetchByIdAndSet($authUser);
            } catch (ServiceNotFoundException $e) {
                // No cache? nothing to invalidate
            }
        }
        return $userDao->fetchById($authUser, $p4Admin);
    }

    /**
     * @inheritDoc
     */
    public function setCookieInformation(array $data): void
    {
        $services = $this->services;
        $path     = $data[self::BASE_URL];
        $isHTTPS  = isset($data[self::HTTPS]);
        $config   = $services->get(ConfigManager::CONFIG);
        $session  = $services->get(self::SESSION);
        // Remember was an option passed from the 'old' login. Not sure if it will
        // be kept so we check to see if we have it
        $remember = isset($data[self::REMEMBER]) ?: null;
        // The 'remember' setting may have changed; ensure the session cookie
        // is set for the appropriate lifetime before we regenerate its id
        $config[self::SESSION] +=
            [self::REMEMBERED_COOKIE_LIFETIME => null, self::COOKIE_LIFETIME => null];
        $session->getConfig()->setStorageOption(
            self::COOKIE_LIFETIME,
            $config[self::SESSION][$remember ? self::REMEMBERED_COOKIE_LIFETIME : self::COOKIE_LIFETIME]
        );
        // Regenerate our id since they logged in; this avoids session fixation and also
        // allows any lifetime changes to take affect.
        // As the session was already started there's a Set-Cookie entry for it
        // and regenerating would normally add a second. to avoid two entries (harmless
        // but off-putting) we first clear all Set-Cookie headers.
        header_remove('Set-Cookie');
        session_regenerate_id(true);
        $strict   = ConfigManager::getValue($config, ConfigManager::SECURITY_HTTPS_STRICT, false);
        $sameSite = ConfigManager::getValue($config, ConfigManager::SESSION_COOKIE_SAMESITE);
        // If the config value for samesite is set to None but connection is not secure, https, reverting config to Lax
        if ($sameSite === ConfigManager::NONE && !$strict && !$isHTTPS) {
            $config[ConfigManager::SESSION][ConfigManager::COOKIE_SAMESITE] = ConfigManager::LAX;
            $services->setService(ConfigManager::CONFIG, $config);
        }
        $cookieOptions = array (
            'expires' => $remember ? time() + 365 * 24 * 60 * 60 : -1,
            'path' => $path,
            'domain' => '',
            'secure' => $strict || $isHTTPS,
            'httponly' => true,
            'samesite' => ConfigManager::getValue($config, ConfigManager::SESSION_COOKIE_SAMESITE)
        );
        if ($remember) {
            // This cookie sticks around for a year. We don't use the session lifetime
            // here as you want the user id to fill in when the session expires (if remember
            // me was checked). if we shared lifetimes with the session, the user id would
            // never be autofilled/remembered when you actually needed it.
            headers_sent() ?: setcookie(self::REMEMBER, $data[self::USERNAME], $cookieOptions);
        } elseif (isset($_COOKIE[self::REMEMBER])) {
            headers_sent() ?: setcookie(self::REMEMBER, null, $cookieOptions);
        }
    }

    /**
     * @inheritDoc
     */
    public function authenticateCandidates(string $userName, string $password, bool $saml = false): array
    {
        $services = $this->services;
        $config   = $services->get(ConfigManager::CONFIG);
        $p4Admin  = $services->get(ConnectionFactory::P4_ADMIN);
        $auth     = $services->get(Services::AUTH);
        $logger   = $services->get(SwarmLogger::SERVICE);
        $authUser = null;
        $adapter  = null;
        $error    = null;
        $ticket   = '';
        // loop through all login candidates, stop on first success
        foreach ($this->getLoginCandidates($userName, $password) as $user) {
            $logger->info(sprintf("[%s]: Trying to log in as [%s]", self::LOG_PREFIX, $user));
            $adapter = new Adapter($user, $password, $p4Admin, $config, $saml);

            try {
                $authResult = $auth->authenticate($adapter);
                // break if we hit a working candidate
                if ($authResult->getCode() === Result::SUCCESS) {
                    $authUser = $user;
                    $ticket   = method_exists($authResult, 'getIdentity') &&
                    is_array($authResult->getIdentity()) && array_key_exists('ticket', $authResult->getIdentity())
                        ? $authResult->getIdentity()['ticket'] : '';
                    $logger->info(sprintf("[%s]: Authenticated user [%s]", self::LOG_PREFIX, $user));
                    break;
                }
            } catch (Exception $e) {
                // we skip any failed accounts; better luck next try :)
                $logger->info(
                    sprintf(
                        "[%s]: Attempt to log in as [%s] failed with [%s]",
                        self::LOG_PREFIX,
                        $user,
                        $e->getMessage()
                    )
                );
                $error = $e->getMessage();
            }
        }
        return [
            self::AUTH_USER => $authUser,
            self::USERNAME  => $userName,
            self::ADAPTER   => $adapter,
            self::TICKET    => $ticket,
            self::IS_VALID  => $authUser !== null,
            self::ERROR     => $error
        ];
    }

    /**
     * @inheritDoc
     */
    public function getLoginCandidates(string $userName, $password = null) : array
    {
        $services = $this->services;
        $config   = $services->get(IConfigDefinition::CONFIG);
        $userDao  = $services->get(IDao::USER_DAO);
        $p4Admin  = $services->get(ConnectionFactory::P4_ADMIN);
        $blocked  = ConfigManager::getValue($config, IConfigDefinition::SECURITY_PREVENT_LOGIN, []);
        $logger   = $services->get(SwarmLogger::SERVICE);

        // If we are passed an email (anything with an @) find all matching accounts.
        // Otherwise, simply fetch the passed id.
        $candidates = [];
        $logger->info(sprintf("[%s]: Getting login candidates for [%s]", self::LOG_PREFIX, $userName));
        if (strpos($userName, '@')) {
            foreach ($userDao->fetchAll([], $p4Admin) as $candidate) {
                $logger->trace(
                    sprintf(
                        "[%s]: Looking for match user email [%s], user id [%s]",
                        self::LOG_PREFIX,
                        $candidate->getEmail(),
                        $candidate->getId()
                    )
                );
                // For email, we always use a case-insensitive comparison
                if (mb_strtolower($candidate->getEmail()) === mb_strtolower($userName)) {
                    $logger->info(
                        sprintf(
                            "[%s]: Match found for email [%s], user id [%s]",
                            self::LOG_PREFIX,
                            $userName,
                            $candidate->getId()
                        )
                    );
                    $candidates[] = $candidate->getId();
                }
            }
        } else {
            try {
                // If we are provided with a user name it will be equal to what the user
                // entered. We want to look up the User value from the spec and use that
                // instead so that for example issues with case sensitivity are handled.
                // For example on a case insensitive server where a user had been created
                // with a value of 'Bruno' logging in as 'bruno', 'Bruno', or 'BRUNO' should
                // all result in 'Bruno' as the value to use for the user connection. That way it
                // should match fields such as author and participants that have been
                // added to reviews. This will also then behave in the way as if an email had
                // been provided
                $filter       = new UserFilter($services);
                $candidates[] = $filter->filter($userName, $password);
            } catch (InvalidArgumentException $e) {
                // Invalid user, auth module will generate the response
                $candidates[] = $userName;
            }
        }
        $caseSensitive = $p4Admin->isCaseSensitive();
        return $caseSensitive ? array_diff($candidates, $blocked) : array_udiff($candidates, $blocked, 'strcasecmp');
    }

    /**
     * @inheritDoc
     */
    public function getSSO(string $serverId = null): string
    {
        $config = $this->services->get(IConfigDefinition::CONFIG);
        $logger = $this->services->get(SwarmLogger::SERVICE);
        $sso    = IConfigDefinition::DISABLED;

        try {
            if ($serverId) {
                try {
                    $logger->debug(
                        sprintf(
                            "[%s]: Get [%s] for server id [%s]",
                            self::LOG_PREFIX,
                            IConfigDefinition::SSO,
                            $serverId
                        )
                    );
                    $sso = ConfigManager::getValue(
                        $config,
                        (IConfigDefinition::P4 . '.' . $serverId . '.' . IConfigDefinition::SSO),
                        IConfigDefinition::DISABLED
                    );
                } catch (ConfigException $e) {
                    $logger->debug(
                        sprintf(
                            "[%s]: Get [%s] for server id [%s]",
                            self::LOG_PREFIX,
                            IConfigDefinition::SSO_ENABLED,
                            $serverId
                        )
                    );
                    $sso = ConfigManager::getValue(
                        $config,
                        (IConfigDefinition::P4 . '.' . $serverId . '.' . IConfigDefinition::SSO_ENABLED),
                        false
                    );
                }
            } else {
                $sso =  ConfigManager::getValue($config, IConfigDefinition::P4_SSO, IConfigDefinition::DISABLED);
            }
            if (is_bool($sso)) {
                $sso = $sso ? IConfigDefinition::ENABLED : IConfigDefinition::DISABLED;
            }
            return $sso;
        } catch (ConfigException $e) {
            if ($e->getCode() === ConfigException::PATH_DOES_NOT_EXIST) {
                // This can be expected
                return $sso;
            }
            throw $e;
        }
    }

    /**
     * @inheritDoc
     */
    public function getLoginReferrer(): ?string
    {
        $referrer = null;
        if ($_REQUEST && isset($_REQUEST[self::RELAY_STATE])) {
            $loginMatch = preg_match('/.*\/api\/v[0-9]+\/login\/saml*/', $_REQUEST[self::RELAY_STATE]);
            if (!$loginMatch) {
                $referrer = $_REQUEST[self::RELAY_STATE];
            }
        }
        return $referrer;
    }

    /**
     * @inheritDoc
     */
    public function getSaml(): string
    {
        $config = $this->services->get(IConfigDefinition::CONFIG);
        try {
            return ConfigManager::getValue(
                $config,
                IConfigDefinition::SAML_ACS_URL
            ) ? IConfigDefinition::ENABLED : IConfigDefinition::DISABLED;
        } catch (ConfigException $exception) {
            if ($exception->getCode() === ConfigException::PATH_DOES_NOT_EXIST) {
                // This can be expected
                return IConfigDefinition::DISABLED;
            }
            throw $exception;
        }
    }
}
