<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.1/2745343
 */

namespace Authentication\Service;

use Application\Config\Services;
use Application\Connection\ConnectionFactory;
use Application\I18n\TranslatorFactory;
use Application\Model\IModelDAO;
use Application\Permissions\Exception\BasicAuthFailedException;
use Application\Permissions\Exception\UnauthorizedException;
use Laminas\Http\Response;
use Laminas\View\Model\JsonModel;
use P4\Connection\Exception\ServiceNotFoundException;
use Authentication\Helper\ILoginHelper;
use Users\Model\IUser;
use Exception;

class BasicLoginAdapter extends AbstractLoginAdapter implements LoginAdapterInterface
{
    /**
     * Login, note this uses the content body and ignores any data provided
     * Login to Swarm using the credentials provided
     * @param mixed $data
     * @return JsonModel
     * @throws ServiceNotFoundException
     * @throws \P4\Exception
     * @throws BasicAuthFailedException|UnauthorizedException
     */
    public function login($data): JsonModel
    {
        $translator = $this->services->get(TranslatorFactory::SERVICE);
        $msg        = [];
        try {
            $session = $this->services->get(ILoginHelper::SESSION);
            $helper  = $this->services->get(ILoginHelper::HELPER_NAME);
            // Clear any existing session data on login, we need to explicitly restart the session
            $session->start();
            $session->getStorage()->clear();
            if (isset($data[ILoginHelper::USERNAME]) && isset($data[ILoginHelper::PASSWORD])) {
                $userName = $data[ILoginHelper::USERNAME];
                $password = $data[ILoginHelper::PASSWORD];
                $p4Admin  = $this->services->get(ConnectionFactory::P4_ADMIN);
                $candAuth = $helper->authenticateCandidates($userName, $password);
                $authUser = $candAuth[ILoginHelper::AUTH_USER];
                if ($candAuth[ILoginHelper::IS_VALID]) {
                    $helper->setCookieInformation($data);
                    $helper->invalidateCache($authUser, $p4Admin);
                } else {
                    try {
                        // verify that the admin connection is still valid if so then the user credentials
                        // provided are incorrect.
                        $userDao = $this->services->get(IModelDAO::USER_DAO);
                        $userDao->fetchById($p4Admin->getUser(), $p4Admin);
                        throw new BasicAuthFailedException(
                            ILoginHelper::TEXT_LOGIN_INCORRECT,
                            Response::STATUS_CODE_401
                        );
                    } catch (Exception $le) {
                        if ($le instanceof BasicAuthFailedException) {
                            throw $le;
                        }
                        throw new ServiceNotFoundException(
                            ILoginHelper::TEXT_SWARM_CONNECTION,
                            Response::STATUS_CODE_500
                        );
                    }
                }
            } else {
                throw new \P4\Exception(ILoginHelper::TEXT_LOGIN_MISSING, Response::STATUS_CODE_400);
            }
            $session->writeClose();
            $msg[] = self::buildMessage(
                ILoginHelper::CODE_LOGIN_SUCCESSFUL,
                $translator->t(ILoginHelper::TEXT_LOGIN_SUCCESSFUL)
            );
            $user  = $this->currentUser($candAuth[ILoginHelper::TICKET]);
            return self::success([IUser::USER => [$user]], $msg);
        } catch (BasicAuthFailedException $error) {
            $msg[] = self::buildMessage(ILoginHelper::CODE_LOGIN_INCORRECT, $translator->t($error->getMessage()));
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_401);
        } catch (UnauthorizedException $error) {
            $msg[] = self::buildMessage(ILoginHelper::CODE_NOT_LOGGED_IN, $translator->t($error->getMessage()));
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_401);
        } catch (ServiceNotFoundException $error) {
            $msg[] = self::buildMessage(ILoginHelper::CODE_SWARM_CONNECTION, $translator->t($error->getMessage()));
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_500);
        } catch (\P4\Exception $error) {
            $msg[] = self::buildMessage(ILoginHelper::CODE_LOGIN_MISSING, $translator->t($error->getMessage()));
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_400);
        } catch (Exception $error) {
            $msg[] = self::buildMessage(ILoginHelper::USER_ERROR, $translator->t($error->getMessage()));
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_404);
        }

        return self::error($msg, $this->getResponse()->getStatusCode());
    }
}
