<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.1/2745343
 */

namespace Application\Connection;

use Application\Cache\AbstractCacheService;
use Application\Config\ConfigException;
use Application\Config\ConfigManager;
use Application\Config\IConfigDefinition;
use Application\Model\ServicesModelTrait;
use P4\Connection\Exception\CommandException;
use P4\Connection\Extension;
use P4\Exception as P4Exception;
use P4\Log\Logger;
use Redis\RedisService;

/**
 * This is to extend the extension library, so we can override the isAuthenticated method to use our own version.
 */
class SwarmExtension extends Extension
{
    use ServicesModelTrait;

    protected $ttl;

    /**
     * Constructs a P4 connection object.
     *
     * @param null $port     optional - the port to connect to.
     * @param null $user     optional - the user to connect as.
     * @param null $client   optional - the client spec to use.
     * @param null $password optional - the password to use.
     * @param null $ticket   optional - a ticket to use.
     * @param null $ipaddr   optional - IP address of user's location
     * @param null $svrname  optional - the user running swarm.
     * @param null $svrpass  optional - the user's password that is running swarm.
     *
     * @throws P4Exception
     */
    public function __construct(
        $port = null,
        $user = null,
        $client = null,
        $password = null,
        $ticket = null,
        $ipaddr = null,
        $svrname = null,
        $svrpass = null
    ) {
        parent::__construct($port, $user, $client, $password, $ticket, $ipaddr, $svrname, $svrpass);
        $services  = ServicesModelTrait::getServices();
        $config    = $services->get(IConfigDefinition::CONFIG);
        $this->ttl = ConfigManager::getValue($config, IConfigDefinition::SESSION_USER_LOGIN_STATUS_CACHE, 10);
    }

    /**
     * Check if the user is authenticated
     *
     * Note: if the user has no password, but one has been set on the connection, we consider that not authenticated.
     *
     * @throws ConfigException
     * @return bool     true if user is authenticated, false otherwise
     */
    public function isAuthenticated(): bool
    {

        // If the ttl has been set in the config to be zero we assume customer doesn't want to use our cache service.
        if ($this->ttl === 0) {
            return parent::isAuthenticated();
        }
        $result = [];
        try {
            $result = (array) $this->fetchAuthentication($this->ttl);
            $this->setMFAStatus(array_key_exists(self::TWOFA, $result) ? $result[self::TWOFA] : null);
        } catch (CommandException $e) {
            return false;
        }

        if (array_key_exists(self::AUTHEDBY, $result) || array_key_exists(self::TICKETEXPIRATION, $result)) {
            Logger::log(Logger::TRACE, "We have found ever AuthedBy or TicketExpiration");
            return true;
        } else {
            Logger::log(Logger::TRACE, "We did not find AuthedBy or TicketExpiration in output");
            try {
                $stringResult = $this->fetchAuthentication($this->ttl, false);
            } catch (CommandException $e) {
                return false;
            }
            Logger::log(Logger::TRACE, "The output of none tag login -s is:" . $stringResult);
            // if a password is not required but one was provided, we should reject that connection on principle
            if (strpos("'login' not necessary, no password set for this user.", $stringResult) !== false
                && (strlen($this->password) || strlen($this->ticket))
            ) {
                return false;
            }
        }
        Logger::log(Logger::TRACE, "We have authenticated user with non tagged output");
        return true;
    }

    /**
     * This is a wrapper function to check if we have the value in our cache or not. Using the traits we will get
     * the cache service and then fetch that data. If it's not present we will call the login -s command and save
     * the returned data from that command output.
     *
     * @param int  $ttl    The Time Live we want the cache to have.
     * @param bool $tagged If we want tagged output for the p4 command.
     * @return array|false|string
     * @throws ConfigException
     */
    protected function fetchAuthentication(int $ttl, bool $tagged = true)
    {
        $result          = [];
        $services        = ServicesModelTrait::getServices();
        $cacheConnection = $services->get(AbstractCacheService::CACHE_SERVICE);
        if ($cacheConnection) {
            $cache = $cacheConnection->getRedisResource();
            Logger::log(Logger::TRACE, "We have cache service to use");
            $index   = $cacheConnection->getNamespace() . "UserSessionData" . RedisService::SEPARATOR .$this->getUser()
            . ($tagged ? RedisService::SEPARATOR ."tagged" : "");
            $session = json_decode($cache->get($index));
            if ($session) {
                Logger::log(
                    Logger::TRACE,
                    "We have cached data for index" . $index . " is:" . json_encode($session, true)
                );
                return $session;
            } else {
                Logger::log(Logger::TRACE, "We do not have cached data for index" . $index);
            }
            $result = $this->runLoginStatus($tagged);
            $data   = json_encode($result);
            $cache->set($index, $data, ['nx', 'ex'=> $ttl]);
        } else {
            Logger::log(Logger::TRACE, "We do not have cache service");
            $result = $this->runLoginStatus($tagged);
        }
        return $result;
    }

    /**
     * This will run the login -s command and depending on the tagged value return array for tagged and string for non
     * tagged output.
     * @param bool $tagged This true or false if we want the output to be tagged.
     * @return array|false|string
     */
    protected function runLoginStatus(bool $tagged = true)
    {
        Logger::log(Logger::TRACE, "Fetching Login data for this user ". $this->getUser());
        $result = $this->run('login', '-s', null, $tagged);
        if ($tagged) {
            Logger::log(Logger::TRACE, "results data 0: ". json_encode($result->getData(0), true));
            $result = $result->getData(0);
        } else {
            Logger::log(Logger::TRACE, "results all data: ". json_encode($result->getData(), true));
            $result = implode($result->getData());
        }
        return $result;
    }
}
