<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.1/2745343
 */

namespace AiAnalysis\Service;

use Laminas\View\Model\JsonModel;
use Laminas\Mvc\Controller\AbstractRestfulController;
use Laminas\Http\Client as HttpClient;
use Laminas\Http\Request;

abstract class AbstractAiAdapter extends AbstractRestfulController
{
    const API_BASE = '/api';
    const ERROR    = 'error';
    const MESSAGES = 'messages';
    const DATA     = 'data';
    const CODE     = 'code';
    const TEXT     = 'text';

    const DATA_DIFF_CONTENT = 'diffContent';

    protected $services = null;

    /**
     * IndexController constructor.
     * @param $services
     */
    public function __construct($services)
    {
        $this->services = $services;
    }
    /**
     * Build a message with the code and message text
     * @param mixed  $code      message code
     * @param string $text      message text
     * @return array with self::CODE => $code and self::TEXT => $text
     */
    public function buildMessage($code, string $text): array
    {
        return [self::CODE => $code, self::TEXT => $text];
    }

    /**
     * Return Success Response
     * @param mixed $data           Data that will be returned
     * @param array $messages       optional messages, defaults to empty array
     * @return JsonModel
     */
    public function success($data, array $messages = []): JsonModel
    {
        return $this->buildResponse($data, $messages);
    }

    /**
     * Return Error Response
     * @param  array    $messages       messages
     * @param  mixed    $errorCode      error code
     * @param  mixed    $data           optional data
     * @return JsonModel
     */
    public function error(array $messages, $errorCode, $data = null): JsonModel
    {
        return $this->buildResponse($data, $messages, $errorCode);
    }

    /**
     * Prepares Json Response
     * @param  array|null   $data           Data that will be returned
     * @param  array        $messages       messages to return
     * @param  int|null     $errorCode      error code
     * @return JsonModel
     */
    private function buildResponse($data, array $messages, $errorCode = null): JsonModel
    {
        $returnResponse = [
            self::ERROR     => $errorCode,
            self::MESSAGES  => $messages,
            self::DATA      => $data
        ];
        return new JsonModel($returnResponse);
    }

    /**
     * Function that make the GET or POST requests
     * @param string $url Url in which we want to make our request to
     * @param bool $fetch Set to true for a GET request, otherwise will do a POST.
     * @param array $args These are the parameter that we pass the GET or POST request to filter the reviews
     * @param array $headers These are the parameter that we pass the GET or POST request to filter the reviews
     * @param string $mimeType mimeType for the given request
     */
    protected function request(
        string $url,
        bool $fetch = false,
        array $args = array(),
        array $headers = [],
        string $mimeType = 'application/json'
    ) {
        $client = new HttpClient;
        $client->setOptions(
            [
            'maxredirects' => 0,
            'timeout' => 30,
            ]
        );
        $client->setUri($url);
        if ($headers) {
            $client->setHeaders($headers);
        }
        $client->setMethod(Request::METHOD_POST);
        $client->setRawBody(json_encode($args));
        $client->setEncType($mimeType);

        $response = $client->dispatch($client->getRequest());
        $response->getHeaders()->addHeaderLine('Content-Type', 'application/json');
        $response->getHeaders()->addHeaderLine('Accept-Encoding', 'gzip,deflate,br');
        $encodedContent  = $response->getContent();
        $responseContent = json_decode($encodedContent);
        return $responseContent->data;
    }
}
