#!/bin/bash

config="${1:-config.properties}"

## Check for perforce
if ! command -v p4 &>/dev/null; then
  echo "The p4 command could not be found, please install or configure the PATH."
  exit
fi

## Check config file
if ! [[ -f "$config" ]]; then
  echo "Unable to find the config file: $config"
  exit
fi
if ! [[ -w "$config" ]]; then
  echo "Unable to write to the config file: $config"
  exit
fi

indent() { sed 's/^/    /'; }

password() {
  __result=$1
  prompt=$2
  password=""
  while IFS= read -p "$prompt" -r -s -n 1 char; do
    if [[ $char == $'\0' ]]; then
      break
    fi
    prompt='*'
    password+="$char"
  done
  echo
  eval $__result="'$password'"
}

check() {
  __code=$?
  __result=$@
  if [ $__code -ne 0 ]; then
    echo
    echo "---- [ ERROR ] -----------------------------------------------------------------"
    echo "$__result" | indent
    echo "--------------------------------------------------------------------------------"
    exit $__code
  fi
}

getConfig() {
  __result=$1
  __key=$2
  value=$(grep "^${__key}" config.properties | sed "s/^${__key}=//")
  eval $__result="'$value'"
}

setConfig() {
  __key=$1
  __value=$2

  if [[ "${OSTYPE//[0-9.]/}" == "darwin" ]]; then
        $(sed -i '' "s|^${__key}.*|${__key}=${__value}|" config.properties)
  else
        $(sed -i "s|^${__key}.*|${__key}=${__value}|" config.properties)
  fi
}

createServer() {
  __extUrl=$1

  echo
  echo "Creating server spec for p4search..."
  getConfig serviceUser "com.perforce.p4search.core.service.p4user"

  read -p "  P4Search server spec [p4search]: " serverID
  serverID=${serverID:-p4search}

  servers=$(p4 -p $p4port -u $p4user -F%serverID% servers | grep ^${serverID}$)
  if [[ $servers =~ $serverID ]]; then
    echo "  Skipping; already exists"
  else
    echo
    echo "Server '$serverID' does not exist!"
    read -r -p "  Create user? [Y/n]: " create
    if [[ $create =~ ^(Yes|yes|Y|y| ) ]] || [[ -z $create ]]; then
      echo
      echo "Creating '${serverID}' server spec..."

      host=$(echo ${__extUrl} | sed -e "s|http.*:\/\/\(.*\):\(.*\)|\1|")
      port=$(echo ${__extUrl} | sed -e "s|http.*:\/\/\(.*\):\(.*\)|\2|")

      fields=$(cat <<-EOM
User: $serviceUser

Address: $host

AllowedAddresses:
  $host:$port
EOM)

      check $((p4 -p $p4port -u $p4user server -o p4search; echo "$fields") | p4 -p $p4port -u $p4user server -i)
      echo "  OK"
    fi
  fi
}

createUser() {
  __use=$1
  __permission=$2
  __type=$3
  __allhost=$4

  echo
  echo "Configuring HelixCore $__use user account..."

  getConfig searchUserDefault "com.perforce.p4search.core.$__use.p4user"
  read -p "  Username [$searchUserDefault]: " searchUser
  searchUser=${searchUser:-$searchUserDefault}
  if [[ $p4user == $searchUser ]]; then
    echo
    echo "You cannot use the current user '$p4user' for $__use user account!"
    exit 1
  fi
  setConfig "com.perforce.p4search.core.$__use.p4user" $searchUser

  password searchPass "  Enter password for $__use: "

  users=$(p4 -p $p4port -u $p4user -F%user% users -a | grep ^${searchUser}$)
  if ! [[ $users =~ $searchUser ]]; then
    echo
    echo "User '$searchUser' does not exist!"
    read -r -p "  Create user? [Y/n]: " create
    if [[ $create =~ ^(Yes|yes|Y|y| ) ]] || [[ -z $create ]]; then
      echo
      echo "Creating '${searchUser}' user account..."
      check $((p4 -p $p4port -u $p4user user -o $searchUser; echo "Type: $__type") | p4 -p $p4port -u $p4user user -f -i)
      echo "  OK"
      echo
      echo "Setting '${searchUser}' password..."
      check $(echo -e "$searchPass\n$searchPass\n" | p4 -p $p4port -u $p4user passwd $searchUser)
      echo "  OK"
    fi
  fi

  echo
  echo "Configuring unlimited ticket group..."
  read -p "  Group name [unlimited]: " unlimited
  unlimited=${unlimited:-unlimited}

  groups=$(p4 -p $p4port -u $p4user groups | grep ^${unlimited}$)
  if ! [[ $groups =~ $unlimited ]]; then
    echo
    echo "Group '$unlimited' does not exist!"
    read -r -p "  Create group? [Y/n]: " create
    if [[ $create =~ ^(Yes|yes|Y|y| ) ]] || [[ -z $create ]]; then
      echo
      echo "Creating '$unlimited' group..."
      check $(p4 -p $p4port -u $p4user group -o $unlimited | sed "s/^Owners:.*/Owners: ${p4user}/" | sed 's/^Timeout:.*/Timeout: unlimited/' | p4 -p $p4port -u $p4user group -i)
      echo "  OK"
    fi
  fi

  members=$(p4 -p $p4port -u $p4user groups -u $searchUser | grep ^${unlimited}$)
  if ! [[ $members =~ $unlimited ]]; then
    echo
    echo "User $searchUser is not a member of $unlimited!"
    read -r -p "  Add user to group? [Y/n]: " create
    if [[ $create =~ ^(Yes|yes|Y|y| ) ]] || [[ -z $create ]]; then
      echo
      echo "Adding $searchUser to group..."
      check $(p4 -p $p4port -u $p4user group -o $unlimited | {
        cat -
        echo "  $searchUser"
      } | p4 -p $p4port -u $p4user group -i)
      echo "  OK"
    fi
  fi

  echo
  echo "Checking '${searchUser}' permissions..."
  level=$(p4 -p $p4port -u $p4user protects -u $searchUser -m)
  if ! [[ $level =~ ($__permission) ]]; then
    echo
    echo "User $searchUser does not have '$__permission' access!"
    read -r -p "  Append '$__permission user $searchUser * //...' permission to protects? [Y/n]: " create
    if [[ $create =~ ^(Yes|yes|Y|y| ) ]] || [[ -z $create ]]; then
      echo
      echo "Adding permission..."
      check $(p4 -p $p4port -u $p4user protect -o | {
        cat -
        echo "  $__permission user $searchUser * //..."
      } | p4 -p $p4port -u $p4user protect -i)
      echo "  OK"
    fi
  fi

  echo
  echo "Generating ${__allhost} '${searchUser}' ticket..."
  check $(echo $searchPass | p4 -p $p4port -u $searchUser login)
  if [[ __allhost == "allhost" ]]; then
    ticket=$(echo $searchPass | p4 -p $p4port -u $searchUser login -p -a | grep -v 'Enter password:')
  else
    ticket=$(echo $searchPass | p4 -p $p4port -u $searchUser login -p | grep -v 'Enter password:')
  fi
  setConfig "com.perforce.p4search.core.$__use.p4ticket" $ticket
}

createExtensionsDepot() {
  extDepots=$(p4 -p $p4port -u $p4user depots -t extension)
  if ! [[ $extDepots =~ "extension" ]]; then
	echo	
	echo "Extensions depot does not exist!"
	read -r -p "  Create extension depot? [Y/n]: " create
    if [[ $create =~ ^(Yes|yes|Y|y| ) ]] || [[ -z $create ]]; then
      echo
      echo "Creating extensions depot..."
      check $(p4 -p $p4port -u $p4user depot -t extension -o .p4-extensions | p4 -p $p4port -u $p4user depot -i )
      echo "  OK"
      echo
    fi
  fi
}

## Connect to Perforce
echo "Connecting to your Perforce Server..."
getConfig p4portDefault "com.perforce.p4search.core.p4port"
read -p "  Perforce Server [$p4portDefault]: " p4port
p4port=${p4port:-$p4portDefault}
setConfig "com.perforce.p4search.core.p4port" $p4port


## Establish trust (for ssl)
if [[ "$p4port" =~ ^ssl* ]]; then
  check $(p4 -p $p4port trust -d)
  fingerprint=$(p4 -p $p4port trust -fy | grep 'fingerprint is' | sed "s/That fingerprint is //")
  echo "  Using fingerprint: $fingerprint"
  setConfig "com.perforce.p4search.core.p4trust" $fingerprint
fi


## Login
read -p "  Superuser account name [super]: " p4user
p4user=${p4user:-super}
password p4pass "  Enter Password: "
check $(echo $p4pass | p4 -p $p4port -u $p4user login 2>&1)
echo "  OK"


## External access
echo
echo "Configure the external URL for P4Search (used by the Perforce Server)..."
getConfig extUrlDefault "com.perforce.p4search.service.external-url"
read -p "  External URL [$extUrlDefault]: " extUrl
extUrl=${extUrl:-$extUrlDefault}
setConfig "com.perforce.p4search.service.external-url" $extUrl
echo "  OK"

## CORs configuration
echo
echo "Configure CORs pattern (Cross-Origin Resource Sharing)..."
getConfig corsDefault "com.perforce.p4search.service.access-control-allow-origin"
if [[ -z $corsDefault ]]; then
  corsDefault=${extUrl}
fi
read -p "  Access control allow origin [$corsDefault]: " cors
cors=${cors:-$corsDefault}
setConfig "com.perforce.p4search.service.access-control-allow-origin" $cors
echo "  OK"

## Check Perforce security level
security=$(p4 -p $p4port -u $p4user counter security)
echo
echo "Security level: $security"

## Create standard users
createUser index super standard allhost

## Create service user
if [ "$security" -gt "4" ]; then
  echo "NOTE: in high security mode you must use a different Perforce user for the 'service' and 'index' users."
  createUser service admin service local
  createServer $extUrl
else
  echo "NOTE: in low security mode you can use the same Perforce user for both 'service' and 'index' users."
  createUser service admin standard local
fi

## Create extensions depot
createExtensionsDepot

## Connect to Elastic
echo
echo "Connecting to your Elastic Server(s)..."
getConfig esHostDefault "com.perforce.p4search.elastic.hosts"
read -p "  ElasticSearch host(s) (comma seperated) [$esHostDefault]: " esHosts
esHosts=${esHosts:-$esHostDefault}
setConfig "com.perforce.p4search.elastic.hosts" $esHosts

IFS=',' read -r esHost string <<< "$esHosts"
devPath=$(echo ${esHost} | sed -e "s|http.*:\/\/\(.*\):\(.*\)|/dev/tcp/\1/\2|")
if ! 2> /dev/null >$devPath; then
  echo
  echo "Elastic host not found. $esHost"
  exit
fi

getConfig esUserDefault "com.perforce.p4search.elastic.user"
read -p "  Elastic username [$esUserDefault]: " esUser
esUser=${esUser:-$esUserDefault}
setConfig "com.perforce.p4search.elastic.user" $esUser

getConfig esPassDefault "com.perforce.p4search.elastic.pass"
password esPass "  Elastic password: "
setConfig "com.perforce.p4search.elastic.pass" $esPass

http_code=$(curl --write-out '%{http_code}' --silent --insecure --output /dev/null --user $esUser:$esPass ${esHost}/_cat/indices)
if [[ "$http_code" -ne 200 ]]; then
  echo
  echo "Failed with: $http_code"
  exit
else
  echo "  Connection OK"
fi

echo ""
echo "--------------------------------------------------------------------------------"
echo "  Configuration finished."
echo ""
echo "  After making configuration changes, you must restart the service:"
echo "    sudo systemctl restart p4search.service"
echo "--------------------------------------------------------------------------------"