<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.1/2745343
 */
namespace Projects\Filter;

use Application\Filter\FormBoolean;
use Application\Filter\StringToId;
use Laminas\Filter\AbstractFilter;
use Laminas\Filter\StringTrim;
use Projects\Model\IProject;

/**
 * A filter to convert branches data to get it ready for validation
 */
class Branches extends AbstractFilter
{
    // Defaults used for branch data that is not set
    const BRANCH_DEFAULTS = [
        IProject::ID                       => null,
        IProject::NAME                     => null,
        IProject::BRANCH_PATHS             => '',
        IProject::MODERATORS               => [],
        IProject::MODERATORS_GROUPS        => [],
        IProject::DEFAULTS                 => [IProject::REVIEWERS => []],
        IProject::WORKFLOW                 => null,
        IProject::RETAIN_DEFAULT_REVIEWERS => false,
        IProject::MINIMUM_UP_VOTES         => null
    ];

    /**
     * Filter each branch to convert any values as required ready for validation. Branches
     * are expected to be an array of arrays
     * - if the value is not an array of arrays returns the value
     * - if the value is valid type then provide defaults and any conversion
     *      - defaults are defined in BRANCH_DEFAULTS
     *      - id is populated based in StringToId from the name
     *      - paths are converted to an array if they are a string with line breaks
     *      - once the paths are an array they are trimmed
     *      - convert any default reviewers to <name> => []
     *      - set workflow to null if it is IWorkflow::NO_WORKFLOW_ID
     * @param mixed $value the 'branches' data
     * @return array|mixed|null
     */
    public function filter($value)
    {
        $toId       = new StringToId();
        $value      = $value === '' ? null : $value;
        $normalized = [];
        if (is_array($value) && in_array(true, array_map('is_array', $value))) {
            $formBoolean = new FormBoolean();
            foreach ((array) $value as $branch) {
                $branch = (array) $branch + self::BRANCH_DEFAULTS;
                $branch = array_intersect_key($branch, self::BRANCH_DEFAULTS);

                $branch[IProject::ID]                       = $toId->filter($branch[IProject::NAME]);
                $branch[IProject::RETAIN_DEFAULT_REVIEWERS] =
                    $formBoolean->filter($branch[IProject::RETAIN_DEFAULT_REVIEWERS]);

                // turn paths text input into an array
                // trim and remove any empty entries
                $paths                          = $branch[IProject::BRANCH_PATHS];
                $paths                          = is_array($paths) ? $paths : preg_split("/[\n\r]+/", $paths);
                $branch[IProject::BRANCH_PATHS] = array_values(array_filter(array_map('trim', $paths), 'strlen'));

                $defaultsFilter             = new Defaults();
                $branch[IProject::DEFAULTS] = $defaultsFilter($branch[IProject::DEFAULTS] ?? []);

                if (isset($branch[IProject::WORKFLOW])) {
                    $stringTrim                 = new StringTrim();
                    $value                      = $stringTrim->filter($branch[IProject::WORKFLOW]);
                    $wfFilter                   = new Workflow();
                    $branch[IProject::WORKFLOW] = $wfFilter->filter($value);
                }
                $normalized[] = $branch;
            }
            return $normalized;
        }
        return $value;
    }
}
