<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.1/2745343
 */

use Api\Controller\IndexController as ApiController;
use Api\IRequest;
use Application\Checker;
use Application\Config\ConfigManager;
use Application\Config\IConfigDefinition;
use Application\Factory\InvokableServiceFactory;
use Application\Model\IModelDAO;
use Laminas\Http\Request;
use Laminas\Router\Http\Method;
use Laminas\Router\Http\Segment;
use Markdown\Settings as MarkdownSettings;
use Projects\Controller\BranchApi;
use Projects\Controller\ProjectApi;
use Projects\Filter\GetProjects;
use Projects\Filter\Project as ProjectFilter;
use Projects\Filter\ProjectV9 as ProjectFilterV9;
use Projects\Controller\IndexController;
use Application\View\Helper\ViewHelperFactory;
use Application\Controller\IndexControllerFactory;
use Projects\Helper\Readme;
use Redis\Model\ProjectDAO;
use Application\Config\Services;
use Projects\Helper\FindAffected;
use Projects\Checker\Project as ProjectChecker;
use Projects\Checker\IProject;
use Projects\Filter\BranchesInput;

return [
    'projects' => [
        IConfigDefinition::MAINLINES       => ['main', 'mainline', 'master', 'trunk'],  // common mainline branch ids
        IConfigDefinition::ADD_ADMIN_ONLY  => null,     // if enabled only admin users can create projects
        IConfigDefinition::ADD_GROUPS_ONLY => null,     // if set, only members of given groups can create projects
        'edit_name_admin_only'         => false,    // if enabled only admin users can edit project name
        'edit_branches_admin_only'     => false,    // if enabled only admin users can add/edit/remove project branches
        'private_by_default'           => false,    // if enabled then new projects will have 'private' option checked
        IConfigDefinition::MAX_README_SIZE => null,     // if enabled this will limit readme.md size. Size in Bytes.
        IConfigDefinition::README_MODE     => MarkdownSettings::ENABLED, // Read me files allow markdown by default, the
                                                                     // parse level is defaulted in
                                                                     // markdown->markdown (safe)
        IConfigDefinition::FETCH => [ConfigManager::MAXIMUM => 0], // 0 === get all projects, set to > 0 when
                                                                   // there are lots
        IConfigDefinition::ALLOW_VIEW_SETTINGS => true,
        IConfigDefinition::RUN_TESTS_ON_UNCHANGED_SUBMIT => false,
        IConfigDefinition::DEFAULT_UI => IConfigDefinition::PREVIEW,
        IConfigDefinition::PERMISSION_CHECK => false, // if enabled, branch path permissions are checked during save
    ],
    'router' => [
        'routes' => [
            'project' => [
                'type' => 'Laminas\Router\Http\Segment',
                'options' => [
                    'route'    => '/project[s]/:project[/]',
                    'defaults' => [
                        'controller' => IndexController::class,
                        'action'     => 'project'
                    ],
                ],
            ],
            'project-overview' => [
                'type' => 'Laminas\Router\Http\Segment',
                'options' => [
                    'route'    => '/projects/:project/overview[/]',
                    'defaults' => [
                        'controller' => IndexController::class,
                        'action'     => 'project'
                    ],
                ],
            ],

            'legacy-edit-project' => [
                'type' => 'Laminas\Router\Http\Segment',
                'options' => [
                    'route'    => '/project[s]/edit/:project[/]',
                    'defaults' => [
                        'controller' => IndexController::class,
                        'action'     => 'edit'
                    ],
                ],
            ],

            'edit-project' => [
                'type' => 'Laminas\Router\Http\Segment',
                'options' => [
                    'route'    => '/projects/:project/settings[/]',
                    'defaults' => [
                        'controller' => IndexController::class,
                        'action'     => 'edit'
                    ],
                ],
            ],
            'delete-project' => [
                'type' => 'Laminas\Router\Http\Segment',
                'options' => [
                    'route'    => '/project[s]/delete/:project[/]',
                    'defaults' => [
                        'controller' => IndexController::class,
                        'action'     => 'delete'
                    ],
                ],
            ],
            'project-activity' => [
                'type' => 'Laminas\Router\Http\Segment',
                'options' => [
                    'route'    => '/project[s]/:project/activity[/]',
                    'defaults' => [
                        'controller' => IndexController::class,
                        'action'     => 'activity'
                    ],
                ],
            ],
            'project-reviews' => [
                'type' => 'Laminas\Router\Http\Segment',
                'options' => [
                    'route'    => '/project[s]/:project/reviews[/]',
                    'defaults' => [
                        'controller' => IndexController::class,
                        'action'     => 'reviews'
                    ],
                ],
            ],
            'project-jobs' => [
                'type' => 'Application\Router\Regex',
                'options' => [
                    'regex'    => '/projects?/(?P<project>.+)/jobs?(/(?P<job>.*))?',
                    'spec'     => '/projects/%project%/jobs/%job%',
                    'defaults' => [
                        'controller' => IndexController::class,
                        'action'     => 'jobs',
                        'job'        => null
                    ],
                ],
            ],
            'project-browse' => [
                'type' => 'Application\Router\Regex',
                'options' => [
                    // We specifically do not want to match on projects/edit/ so that we do not
                    // interfere with the legacy-edit-project endpoint if we happen to have projects
                    // named files|view|download|changes
                    'regex'    => '/projects?/(?P<project>(\b(?!edit)\b\S+))/'
                                . '(?P<mode>(files|view|download|changes))(/(?P<path>.*))?',
                    'spec'     => '/projects/%project%/%mode%/%path%',
                    'defaults' => [
                        'controller' => IndexController::class,
                        'action'     => 'browse',
                        'path'       => null
                    ],
                ],
            ],
            'project-archive' => [
                'type' => 'Application\Router\Regex',
                'options' => [
                    'regex'    => '/projects?/(?P<project>[^/]+)/archives?/(?P<path>.+)\.zip',
                    'spec'     => '/projects/%project%/archives/%path%.zip',
                    'defaults' => [
                        'controller' => IndexController::class,
                        'action'     => 'archive'
                    ],
                ],
            ],
            'projects' => [
                'type' => 'Laminas\Router\Http\Segment',
                'options' => [
                    'route'    => '/projects[/]',
                    'defaults' => [
                        'controller' => IndexController::class,
                        'action'     => 'projects'
                    ],
                ],
            ],
            'api' => [
                'type' => 'literal',
                'options' => [
                    'route' => ApiController::API_BASE,
                ],
                'may_terminate' => false,
                'child_routes' => [
                    'api-projects' => [
                        'type' => Segment::class,
                        'options' => [
                            'route' => '/:version/projects',
                            'constraints' => [IRequest::VERSION => 'v1[0-1]'],
                            'defaults' => [
                                'controller' => ProjectApi::class
                            ],
                        ],
                        'child_routes' => [
                            'all-projects' => [
                                'type' => Method::class,
                                'options' => [
                                    'verb' => implode(',', [Request::METHOD_POST, Request::METHOD_GET]),
                                ]
                            ],
                            'project-id' => [
                                'type' => Segment::class,
                                'options' => [
                                    'route' => '/:id',
                                ],
                                'child_routes' => [
                                    'project' => [
                                        'type' => Method::class,
                                        'options' => [
                                            'verb' => implode(
                                                ',',
                                                [
                                                    Request::METHOD_PATCH,
                                                    Request::METHOD_GET,
                                                    Request::METHOD_PUT,
                                                    Request::METHOD_DELETE
                                                ]
                                            ),
                                        ],
                                    ],
                                    'project-undelete' => [
                                        'type' => Segment::class,
                                        'options' => [
                                            'route' => '/undelete',
                                        ],
                                        'child_routes' => [
                                            'undelete' => [
                                                'type' => Method::class,
                                                'options' => [
                                                    'verb' => Request::METHOD_POST,
                                                    'defaults' => [
                                                        'action' => 'undelete'
                                                    ],
                                                ]
                                            ]
                                        ]
                                    ],
                                    'branches' => [
                                        'type' => Segment::class,
                                        'options' => [
                                            'route' => '/branches',
                                            'defaults' => [
                                                'controller' => BranchApi::class
                                            ],
                                        ],
                                        'child_routes' => [
                                            'all-branches' => [
                                                'type' => Method::class,
                                                'options' => [
                                                    'verb' => implode(',', [Request::METHOD_POST, Request::METHOD_GET]),
                                                ]
                                            ],
                                            'branch-id' => [
                                                'type' => Segment::class,
                                                'options' => [
                                                    'route' => '/:'.BranchAPI::IDENTIFIER,
                                                ],
                                                'child_routes' => [
                                                    'branch' => [
                                                        'type' => Method::class,
                                                        'options' => [
                                                            'verb' => implode(
                                                                ',',
                                                                [
                                                                    Request::METHOD_PATCH,
                                                                    Request::METHOD_GET,
                                                                    Request::METHOD_PUT,
                                                                    Request::METHOD_DELETE
                                                                ]
                                                            ),
                                                        ],
                                                    ],
                                                ],
                                            ],
                                        ],
                                    ],
                                    'jobs' => [
                                        'type' => Segment::class,
                                        'options' => [
                                            'route' => '/jobs',
                                        ],
                                        'child_routes' => [
                                            'get-jobs' => [
                                                'type' => Method::class,
                                                'options' => [
                                                    'verb' => Request::METHOD_GET,
                                                    'defaults' => [
                                                        'action' => 'projectJobs'
                                                    ],
                                                ],
                                            ],
                                        ],
                                    ],
                                    'followers' => [
                                        'type' => Segment::class,
                                        'options' => [
                                            'route' => '/followers',
                                        ],
                                        'child_routes' => [
                                            'get-jobs' => [
                                                'type' => Method::class,
                                                'options' => [
                                                    'verb' => implode(
                                                        ',',
                                                        [
                                                            Request::METHOD_DELETE,
                                                            Request::METHOD_GET,
                                                            Request::METHOD_POST,
                                                        ]
                                                    ),
                                                    'defaults' => [
                                                        'action' => 'projectFollowers'
                                                    ],
                                                ],
                                            ],
                                        ],
                                    ],
                                ],
                            ],
                        ],
                    ],
                ],
            ],
        ],
    ],
    'controllers' => [
        'factories' => [
            IndexController::class => IndexControllerFactory::class,
            ProjectApi::class      => IndexControllerFactory::class,
            BranchApi::class          => IndexControllerFactory::class,
        ],
    ],
    'service_manager' => [
        'aliases' => [
            IModelDAO::PROJECT_DAO       => ProjectDAO::class,
            Services::AFFECTED_PROJECTS  => FindAffected::class,
            Services::GET_PROJECT_README => Readme::class,
            Services::GET_PROJECTS_FILTER => GetProjects::class,
            Services::V9_PROJECTS_FILTER=> ProjectFilterV9::class,
            Services::PROJECTS_FILTER => ProjectFilter::class,
            Services::BRANCHES_FILTER => BranchesInput::class
        ],
        'factories' => array_fill_keys(
            [
                ProjectDAO::class,
                ProjectFilterV9::class,
                ProjectFilter::class,
                FindAffected::class,
                Readme::class,
                GetProjects::class,
                ProjectChecker::class,
                BranchesInput::class
            ],
            InvokableServiceFactory::class
        )
    ],
    'view_manager' => [
        'template_map' => [
            'projects/index/project'    => __DIR__ . '/../view/projects/index/project.phtml',
            'projects/index/add'        => __DIR__ . '/../view/projects/index/add.phtml',
            'projects/index/activity'   => __DIR__ . '/../view/projects/index/activity.phtml'
        ],
        'template_path_stack'   => [
            __DIR__ . '/../view',
        ],
    ],
    'view_helpers' => [
        'factories' => array_fill_keys(
            [
                ViewHelperFactory::PROJECT_LIST,
                ViewHelperFactory::PROJECT_SIDEBAR,
            ],
            ViewHelperFactory::class
        )
    ],
    'menu_helpers' => [
        'overview' => [
            'class'    => '\Projects\Menu\Helper\ProjectContextMenuHelper',
            'priority' => 100
        ],
        'projects' => ['priority' => 140],
        'settings' => [
            'class' => '\Projects\Menu\Helper\ProjectContextMenuHelper',
            'priority' => 200
        ]
    ],
    /**
     * Map a check names to be handled by a project checker
     * @see ConfigCheck
     */
    Checker::CHECKERS => [
        array_fill_keys(
            [
                IProject::PROJECT_ADD,
                IProject::PROJECT_EDIT,
                IProject::PROJECT_NAME_ADMIN_ONLY,
                IProject::BRANCHES_ADMIN_ONLY
            ],
            ProjectChecker::class
        )
    ][0]
];
