<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.1/2745343
 */

namespace Files\Service;

use Application\Service\P4Command;
use P4\Command\IDescribe;
use P4\Connection\CommandResult;
use P4\Connection\ConnectionInterface;

/**
 * Class File
 * @package Files\Service
 */
class File extends P4Command
{
    const DIFF2_COMMAND     = 'diff2';
    const FSTAT_COMMAND     = 'fstat';
    const ATTRIBUTE_COMMAND = 'attribute';

    /**
     * Run diff2
     * @param ConnectionInterface   $connection       the connection
     * @param array                 $options          the options
     * @param array                 $files            files
     * @return CommandResult
     */
    public function diff2(ConnectionInterface $connection, array $options, array $files)
    {
        return $this->run($connection, self::DIFF2_COMMAND, $options, $files);
    }

    /**
     * Run 'p4 attribute' on a file. For example to set an attribute on a submitted file with propagation
     *
     * $fileService->attribute($this->p4, 'name', 'value', '//depot/myFile.txt', ['flags' => ['-f', '-p']])
     *
     * or for multiple files at once
     *
     * $fileService->attribute(
     *      $this->p4, 'name', 'value', ['//depot/file1.txt', //depot/file2.txt], ['flags' => ['-f', '-p']]
     * )
     *
     * To remove the attribute the value can be omitted
     *
     * @param ConnectionInterface   $connection the connection
     * @param string                $name       the name of the attribute
     * @param mixed                 $value      the value of the attribute
     * @param string|array          $file       depot file(s). Multiple files can be specified as an array to set
     *                                          the same attribute on all
     * @param array                 $options    the options for the command
     * @return CommandResult
     */
    public function attribute(
        ConnectionInterface $connection,
        string $name,
        $value,
        $file,
        array $options = []
    ) : CommandResult {
        $args = ['-n', $name];
        if ($value) {
            array_push($args, '-v', $value);
        }
        $args = array_merge($args, (array)$file);
        return $this->run($connection, self::ATTRIBUTE_COMMAND, $options, $args);
    }

    /**
     * Run fstat
     * @param ConnectionInterface   $connection       the connection
     * @param array                 $options          the options
     * @param string                $file             file
     * @return CommandResult
     */
    public function fstat($connection, array $options, string $file)
    {
        return $this->run($connection, self::FSTAT_COMMAND, $options, [$file]);
    }

    /**
     * Determine if the type value in the fstat output indicates a ktext file
     * @param mixed         $fstatOutput        the fstat output
     * @return false|int 1 if the type does indicate ktext, 0 if not, false if an error
     */
    public function isKText($fstatOutput)
    {
        // ktext filetypes include things like: ktext, text+ko, text+mko, kxtext, etc.
        return $fstatOutput
            && isset($fstatOutput[IDescribe::TYPE])
            && preg_match('/kx?text|.+\+.*k/i', $fstatOutput[IDescribe::TYPE]);
    }
}
