<?php
/**
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.1/2745343
 */

namespace Authentication\Helper;

use Application\Config\ConfigException;
use P4\Connection\ConnectionInterface;
use P4\Spec\PluralAbstract;
use Laminas\Http\Request;
use Laminas\Http\Response;
use Laminas\View\Model\JsonModel;

/**
 * Interface IHelper describing the responsibilities of a service to help with authentication
 * @package Users\Authentication
 */
interface ILoginHelper
{
    const HELPER_NAME                = 'loginHelper';
    const REMEMBER                   = 'remember';
    const SESSION                    = 'session';
    const USERNAME                   = 'username';
    const USER                       = 'user';
    const PASSWORD                   = 'password';
    const ADAPTER                    = 'adapter';
    const ERROR                      = 'error';
    const IS_VALID                   = 'isValid';
    const AUTH_USER                  = 'authUser';
    const BASE_URL                   = 'baseURL';
    const TICKET                     = 'ticket';
    const PROTOCOL                   = 'protocol';
    const HTTPS                      = 'https';
    const URL                        = 'url';
    const PARAM_REDIRECT             = 'redirect';
    const RELAY_STATE                = 'RelayState';
    const SAML_RESPONSE              = 'SAMLResponse';
    const COOKIE_LIFETIME            = 'cookie_lifetime';
    const LOG_ID                     = 'Authentication:LoginHelper:: ';
    const METHOD                     = 'method';
    const REMEMBERED_COOKIE_LIFETIME = 'remembered_cookie_lifetime';

    const CODE_LOGIN_SUCCESSFUL  = 'user-login-successful';
    const TEXT_LOGIN_SUCCESSFUL  = 'User logged in.';
    const CODE_LOGIN_INCORRECT   = 'user-login-incorrect';
    const TEXT_LOGIN_INCORRECT   = 'User name or password incorrect.';
    const CODE_SWARM_CONNECTION  = 'swarm-connection-error';
    const TEXT_SWARM_CONNECTION  = "An error has occurred with the P4 admin connection '[%s]'";
    const CODE_LOGIN_MISSING     = 'user-login-missing-data';
    const TEXT_LOGIN_MISSING     = 'User name and password must be provided.';
    const CODE_LOGOUT_ERROR      = 'user-logout-error';
    const CODE_LOGOUT_SUCCESSFUL = 'user-logged-out';
    const TEXT_LOGOUT_SUCCESSFUL = 'Successful Logout.';
    const CODE_NOT_LOGGED_IN     = 'user-not-logged-in';
    const TEXT_NOT_LOGGED_IN     = 'User not logged in';
    const CODE_REQUIRE_LOGIN     = 'user-require-login';
    const TEXT_REQUIRE_LOGIN     = 'Required to login';
    const USER_ERROR             = 'user-error';

    const SERVICE_P4_USER_NOT_CREATED = "Service with name \"p4_user\" could not be created";
    const CODE_SSO_NOT_CONFIGURED     = 'sso-not-configured';
    const TEXT_SSO_NOT_CONFIGURED     = 'SSO is not configued.';

    /**
     * Get authenticated user object and invalidate user cache if
     * authenticated user is not in cache - this most likely means
     * that user has been added to Perforce but the form-commit
     * user trigger was not fired
     * @param string    $authUser the user identifier
     * @param ConnectionInterface $p4Admin admin connection
     */
    public function invalidateCache(string $authUser, ConnectionInterface $p4Admin);

    /**
     * Attempts to authenticate each candidate in turn breaking at the first
     * success.
     * @param string $userName the username
     * @param string $password the password
     * @param bool   $saml whether to use Saml
     * @return array results
     * @throws ConfigException
     * @see getLoginCandidates()
     */
    public function authenticateCandidates(string $userName, string $password, bool $saml = false): array;

    /**
     * Determines and sets cookie information for a successful login.
     * @param array $data containing details relevant to setting cookies
     * @throws ConfigException
     */
    public function setCookieInformation(array $data): void;

    /**
     * Gets a list of login candidates. In most cases it will simply be the user name. If the user name
     * happens to represent an email address (contains '@') then the list returned will contain all the
     * user ids that match that email address. Any user names that are blocked by configuration settings
     * are filtered out. Candidates returned will be the values on the user spec that have been found
     * based on the user name or email provided.
     * @param string    $userName   user name provided to the request
     * @return array    list of candidate users
     * @throws ConfigException
     */
    public function getLoginCandidates(string $userName): array;

    /**
     * Checks the server configuration to see if SSO is enabled.
     * @param string|null $serverId the server id to test.
     * @return string (enabled|disabled|optional) checks sso if present else sso_enabled server configuration
     * @throws ConfigException
     */
    public function getSSO(string $serverId = null): string;

    /**
     * Checks the configuration to see if SAML is enabled.
     * @return string (enabled|disabled)
     * @throws ConfigException
     */
    public function getSaml(): string;

    /**
     * Gets the referrer that initiated the login request. If that referrer is
     * the actual login api call then null is returned to let the caller decide
     * where to go to avoid a potential looping call between Swarm and an IDP.
     * It will return string the value of the referrer if set and not the API call path, otherwise null
     * @return string|null
     */
    public function getLoginReferrer(): ?string;
}
