<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.1/2745343
 */

namespace AiAnalysis\Service;

use AiAnalysis\Filter\IAiAnalysis;
use AiAnalysis\Helper\IAiAnalysisHelper;
use Application\Config\ConfigException;
use Application\Config\ConfigManager;
use Application\Config\IConfigDefinition;
use Application\Log\SwarmLogger;
use Laminas\Http\Response;
use OpenAI;

class GenericAIAdapter extends AbstractAiAdapter implements AiServiceInterface
{

    const LOG_PREFIX = GenericAIAdapter::class;

    /**
     * performAI, note this uses the content body and ignores any data provided
     * Login to Swarm using the credentials provided
     *
     * @param mixed $data
     * @return array
     */
    public function executeAIRequest(mixed $data): array
    {
        $logger = $this->services->get(SwarmLogger::SERVICE);
        try {
            $fileName        = ($data[IAiAnalysis::FILE_NAME]) ?
                'of the file: ' . $data[IAiAnalysis::FILE_NAME] . ' ' : ' ';
            $content         = $data[IConfigDefinition::AI_PACKAGE_TYPE] . $fileName .
                $data[IAiAnalysisHelper::CONTENT_TO_ANALYZE];
            $args['content'] = $content;
            $result          = $this->request($this->getApiUrl(), false, $args);
            $logger->trace(sprintf("[%s]: Content generation successful ", self::LOG_PREFIX));
            return [IAiAnalysisHelper::CONTENT_GENERATED => $result->choices[0]->message->content, 'error' => null];
        } catch (\Exception | \Throwable $exception) {
            $logger->debug(
                sprintf(
                    "[%s]: Error occurred at Generic AI Adapter %s",
                    self::LOG_PREFIX,
                    $exception->getMessage()
                )
            );
            $statusCode = $exception->getCode();
            return [
                IAiAnalysisHelper::CONTENT_GENERATED => null,
                'error' => $exception->getMessage(),
                'code' => $statusCode === 0 ? Response::STATUS_CODE_500 : $statusCode,
            ];
        }
    }

    /**
     * This method will fetch the API key from config, required to execute the generic AI request
     * @throws ConfigException
     * @return string
     */
    private function getApiUrl() : string
    {
        $config = $this->services->get(IConfigDefinition::CONFIG);
        return ConfigManager::getValue($config, IConfigDefinition::AI_REVIEW_AI_VENDORS_AI_MODEL1_API_END_POINT);
    }
}
