<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.1/2745343
 */
namespace Projects\Validator;

use Application\Validator\AbstractValidator;
use Application\Validator\IsBool;
use Projects\Model\IProject;
use TestIntegration\Filter\EncodingValidator;

/**
 * Validator for test automation fields on a legacy project
 */
class TestAutomation extends AbstractValidator
{
    protected $messageTemplates = [];

    public function __construct($options = null)
    {
        $this->messageTemplates = [
            IProject::TESTS_URL => $this->tGen('URL for tests must be a string.'),
            IProject::TESTS_ENABLED => $this->tGen('URL for tests must be provided if tests are enabled.'),
            IProject::TESTS_POST_BODY => $this->tGen('POST Body for tests must be a string.'),
            IProject::TESTS_POST_FORMAT => $this->tGen('POST data for tests must be URL or JSON encoded.'),
            IProject::TESTS_POST_BODY . 'Content' =>
                $this->tGen('POST data expected to be %value% encoded, but could not be decoded.')
        ];
        parent::__construct($options);
    }

    /**
     * If set URL should be a string
     * @param $value
     * @return bool
     */
    protected function validateUrl($value): bool
    {
        if (isset($value[IProject::TESTS_URL]) && !is_string($value[IProject::TESTS_URL])) {
            $this->error(IProject::TESTS_URL);
            return false;
        }
        return true;
    }

    /**
     * If enabled is true URL should be present
     * @param $value
     * @return bool
     */
    protected function validateEnabled($value): bool
    {
        $enabled       = $value[IProject::TESTS_ENABLED] ?? false;
        $boolValidator = new IsBool();
        if (!$boolValidator->isValid($enabled)) {
            $this->abstractOptions['messages'][IProject::TESTS_ENABLED] = $boolValidator->getMessages();
            return false;
        } elseif ($enabled && !strlen($value[IProject::TESTS_URL])) {
            $this->error(IProject::TESTS_ENABLED);
            return false;
        }
        return true;
    }

    /**
     * If set the body should be a string
     * @param $value
     * @return bool
     */
    protected function validateBody($value): bool
    {
        if (isset($value[IProject::TESTS_POST_BODY]) && !is_string($value[IProject::TESTS_POST_BODY])) {
            $this->error(IProject::TESTS_POST_BODY);
            return false;
        }
        return true;
    }

    /**
     * Format should be 'url' or 'json' (case insensitive)
     * @param $value
     * @return bool
     */
    protected function validateFormat($value): bool
    {
        $format = $value[IProject::TESTS_POST_FORMAT];
        // We could have used ArrayValuesValidator here, but we do not want to
        // introduce more technical debt around the use of translator
        if (strcasecmp($format, EncodingValidator::URL) !== 0
            && strcasecmp($format, EncodingValidator::JSON) !== 0) {
            $this->error(IProject::TESTS_POST_FORMAT);
            return false;
        }
        return true;
    }

    /**
     * Body content should be able to be parsed correctly for the format given
     * @param $value
     * @return bool
     */
    protected function validateBodyContent($value): bool
    {
        $format = $value[IProject::TESTS_POST_FORMAT];
        $body   = $value[IProject::TESTS_POST_BODY] ?? '';
        if (strcasecmp($format, EncodingValidator::URL) === 0) {
            parse_str($body, $data);
            if (strlen($body) && !count($data)) {
                $this->error(IProject::TESTS_POST_BODY . 'Content', $value[IProject::TESTS_POST_FORMAT]);
                return false;
            }
        } else {
            $data = @json_decode($body, true);
            if (strlen($body) && is_null($data)) {
                $this->error(IProject::TESTS_POST_BODY . 'Content', $value[IProject::TESTS_POST_FORMAT]);
                return false;
            }
        }
        return true;
    }

    /**
     * Validate the value of tests on a legacy project
     * @param $value
     * @return bool
     */
    public function isValid($value): bool
    {
        return $this->validateUrl($value)
            && $this->validateEnabled($value)
            && $this->validateBody($value)
            && $this->validateFormat($value)
            && $this->validateBodyContent($value);
    }
}
