<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.1/2745343
 */

namespace Application\Log;

use Application\Config\ConfigManager;
use Application\Config\IConfigDefinition as ICDef;
use Application\Factory\InvokableService;
use Interop\Container\ContainerInterface;
use Application\Log\Writer\NullWriter;
use Laminas\Log\Writer\Stream;
use P4\Log\DynamicLogger;
use Laminas\Log\Processor\ReferenceId;
use Laminas\Log\Logger as LaminasLogger;
use Application\Config\ConfigException;
use Queue\Manager;

/**
 * Implementation of a logger for Swarm
 * @package Application\Log
 */
class SwarmLogger extends DynamicLogger implements InvokableService
{
    const SERVICE       = 'logger';
    protected $services = null;

    /**
     * @inheritDoc
     * @throws ConfigException
     */
    public function __construct(ContainerInterface $services, array $options = null)
    {
        parent::__construct(
            [
                'priorities' => [
                    [
                        'level' => 8,
                        'name' => 'TRACE'
                    ]
                ]
            ]
        );
        $this->services = $services;

        $config   = $services->get(ICDef::CONFIG);
        $file     = self::getLogFileFromConfig($config);
        $priority = ConfigManager::getValue($config, ICDef::LOG_PRIORITY, null);
        $refId    = ConfigManager::getValue($config, ICDef::LOG_REFERENCE_ID, false);

        // if a file was specified but doesn't exist attempt to create
        // it (unless we are running on the command line).
        // for cli usage we don't want to risk the log being owned by
        // a user other than the web-server so we won't touch it here.
        if ($file && !file_exists($file) && is_writeable(dirname($file)) && php_sapi_name() !== 'cli') {
            touch($file);
        }

        // if a writable file was specified use it, otherwise just use null
        if ($file && is_writable($file)) {
            $writer = new Stream($file);
            if ($priority) {
                $writer->addFilter((int) $priority);
            }
            $this->addWriter($writer);
        } else {
            $this->addWriter(new NullWriter);
        }
        // If we want to trace the apache process we can enable this to help track which
        // worker is doing the work.
        if ($refId) {
            $processor = new ReferenceId();
            $this->addProcessor($processor);
        }
        // register a custom error handler; we can not use the logger's as
        // it would log 'context' which gets vastly too noisy
        set_error_handler(
            function ($level, $message, $file, $line) {
                if (error_reporting() & $level) {
                    $map = LaminasLogger::$errorPriorityMap;
                    $this->log(
                        isset($map[$level]) ? $map[$level] : $this::INFO,
                        $message,
                        [
                            'errno'   => $level,
                            'file'    => $file,
                            'line'    => $line
                        ]
                    );
                }
                return false;
            }
        );
    }

    /**
     * Extending the existing log function to include if we are running a worker to include the slot number.
     *
     * we are merging the request_uri from server request and stripping the query out.
     * If the request is coming in from /queue/worker then we should get the worker slot and add that to the
     * extra field, so we can track which worker is doing this.
     * @param  integer|null         $priority  Priority of message
     * @param  string               $message   Message to log
     * @param  array|Traversable    $extras    Extra information to log in event
     * @return SwarmLogger|void
     */
    public function log($priority, $message, $extra = [])
    {
        // This is needed to get around tests not having uri requests.
        $isTest = defined('IS_TEST');
        if (!$isTest) {
            $extra += [
                "request" => $_SERVER['REQUEST_URI'], "pid" => getmypid()
            ];
            if (strtok($extra["request"], "?") === "/queue/worker") {
                $manager = $this->services->get(Manager::SERVICE);
                $slot    = $manager->getSlot();
                $task    = $manager->getTask();

                $extra["worker"] = $slot;
                $extra["task"]   = $task;
            }

            $extra["REMOTE_ADDR"] = $_SERVER['REMOTE_ADDR'];
            if (isset($_SERVER['HTTP_X_FORWARDED_FOR'])) {
                $extra["HTTP_X_FORWARDED_FOR"] = $_SERVER['HTTP_X_FORWARDED_FOR'];
            }
        }
        parent::log($priority, $message, $extra);
    }

    /**
     * Gets the lofg file value from configuration prefixed with the DATA_PATH
     * @throws ConfigException
     */
    public static function getLogFileFromConfig($config) : string
    {
        $configFile = ConfigManager::getValue($config, ICDef::LOG_FILE);
        return DATA_PATH . (substr($configFile, 0) === '/' ? $configFile : '/' . $configFile);
    }

    /**
     * Builds a logger that does writes to a NullWriter. Useful for dispatch tests
     * that get the contents of stdout and do not want log messages.
     * @return DynamicLogger
     */
    public static function buildNullWriter()
    {
        $logger = self::buildDynamicLogger();
        $logger->addWriter(new NullWriter);
        return $logger;
    }

    /**
     * Builds a dynamic logger with default values.
     * @return DynamicLogger
     */
    private static function buildDynamicLogger()
    {
        return new DynamicLogger(
            [
                'priorities' => [
                    [
                        'level' => 8,
                        'name' => 'TRACE'
                    ]
                ]
            ]
        );
    }
}
