<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.1/2745343
 */

namespace AiAnalysis\Controller;

use AiAnalysis\Filter\IAiAnalysis;
use AiAnalysis\Helper\IAiAnalysisHelper;
use AiAnalysis\Model\AiAnalysisDAO;
use AiAnalysis\Model\AiReviewSummary;
use AiAnalysis\Service\AbstractAiAdapter;
use Api\Controller\AbstractRestfulController;
use Application\Checker;
use Application\Config\ConfigException;
use Application\Config\ConfigManager;
use Application\Config\IConfigDefinition;
use Application\Config\IDao;
use Application\Config\Services;
use Application\Connection\ConnectionFactory;
use Application\Filter\FilterException;
use Application\I18n\TranslatorFactory;
use Application\Log\SwarmLogger;
use Application\Permissions\Exception\ForbiddenException;
use InvalidArgumentException;
use AiAnalysis\Model\IAiReviewSummary;
use Application\Permissions\Exception\UnauthorizedException;
use Application\Permissions\IPermissions;
use Record\Exception\NotFoundException;
use Record\Exception\NotFoundException as RecordNotFoundException;
use Exception;
use Laminas\Http\Request;
use Laminas\Http\Response;
use Laminas\View\Model\JsonModel;
use OpenAI\Exceptions\ErrorException;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\NotFoundExceptionInterface;
use Reviews\Filter\IVersion;
use Record\Key\AbstractKey;
use stdClass;

/**
 * Api controller to manage AI Analysis
 * Class AiAnalysisApi
 * @package AiAnalysis\Controller
 */

class AiAnalysisApi extends AbstractRestfulController
{
    const LOG_PREFIX              = AiAnalysisApi::class;
    const DATA_AI_SUMMARY         = 'aiSummary';
    const DATA_AI_PACKAGE_DETAILS = 'aiPackageDetails';
    const DEFAULT_PACKAGE_ID      = 1;
    /**
     * Analyze Code and Generate AI Summary for Swarm
     *
     * This function triggers the AI service to analyze the provided code or content and generate
     * a summary specific to Swarm. It prepares the necessary data by extracting configurations
     * (like API key, AI model, etc.) and retrieving the content to be analyzed. Once the data
     * is prepared, it sends the request to the AI vendor's API to perform the analysis.
     *
     * The result from the AI vendor, typically a summary or analysis, is returned as a
     * `JsonModel` response, which includes the AI-generated insights.
     *
     * @return  JsonModel
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    public function analyzeCodeAction(): JsonModel
    {
        $aiReviewSummary = '';
        $logger          = $this->services->get(SwarmLogger::SERVICE);
        try {
            $this->checkUserAuthentication();
            $data = $this->getData();
            $this->checkInputCharacterLimits($data[IAiAnalysis::DIFF_CONTENT]);
            $translator  = $this->services->get(TranslatorFactory::SERVICE);
            $p4Admin     = $this->services->get(ConnectionFactory::P4_ADMIN);
            $dao         = $this->services->get(IDao::REVIEW_DAO);
            $review      = $dao->fetchById($data[IAiReviewSummary::REVIEW_ID], $p4Admin);
            $headVersion = $review->getHeadVersion();

            // Getting version from input, default 0 if fromVersion is skipped,default latest version in case toVersion
            $data[IAiReviewSummary::FROM_VERSION] = $data[IAiReviewSummary::FROM_VERSION] ?? 0;
            $data[IAiReviewSummary::TO_VERSION]   = $data[IAiReviewSummary::TO_VERSION] ?? $headVersion;

            // Filtering request data
            $filter = $this->services->get(IAiAnalysis::NAME);
            $filter->setData($data);
            if (!$filter->isValid()) {
                return $this->handleValidationErrors($filter);
            }

            // Validation filter applied
            //'from' must range between -1 and head - 1,
            //'to' must be between 1 and head,
            //'from' should be less than 'to'
            $versionFilter = $this->services->build(
                IVersion::VERSION_FILTER,
                [
                    IVersion::MAX_VERSION => $headVersion,
                    IVersion::MAX_FROM => $data[IAiReviewSummary::TO_VERSION],
                    IVersion::IS_FROM_INCLUSIVE => false
                ]
            );
            $versionFilter->setData(
                [
                    IVersion::FROM => $data[IAiReviewSummary::FROM_VERSION],
                    IVersion::TO => $data[IAiReviewSummary::TO_VERSION]
                ]
            );
            if (!$versionFilter->isValid()) {
                throw new FilterException($versionFilter);
            }

            $aiPackage = $this->getAiPackage($data);
            if (!$aiPackage) {
                return $this->handleError(
                    sprintf(
                        "[%s]: Failed to load the ai package. Please set the default package at config",
                        self::LOG_PREFIX
                    ),
                    Response::STATUS_CODE_500,
                    $translator->t('Failed to load the ai package')
                );
            }
            /** @var AiAnalysisDAO $aiDAO */
            $aiDAO = $this->services->get(IDao::AI_ANALYSIS_DAO);
            $aiDAO->checkRestrictedAccess(IAiReviewSummary::REVIEW . '/' . $data[IAiReviewSummary::REVIEW_ID]);
            $aiDAO->validateFile(
                $data[IAiReviewSummary::FILE_ID],
                $data[IAiReviewSummary::REVIEW_ID],
                $data[IAiReviewSummary::TO_VERSION]
            );

            /** @var AbstractAiAdapter $aiVendorService */
            $aiVendorService = $this->services->get($aiPackage[IConfigDefinition::AI_VENDOR]);
            $logger->info(
                sprintf(
                    "[%s]: AI service requested for review '%s' with package '%s' from vendor '%s'",
                    self::LOG_PREFIX,
                    $data[IAiAnalysis::REVIEW_ID],
                    $aiPackage[IConfigDefinition::AI_PACKAGE_KEY],
                    $aiPackage[IConfigDefinition::AI_VENDOR]
                )
            );

            if ($data[AbstractAiAdapter::DATA_DIFF_CONTENT]) {
                // Fetch the existing record from db based on id
                $aiReviewSummary = isset($data[IAiReviewSummary::ID]) ?
                    $this->fetchResultIfAvailable($data[IAiReviewSummary::ID]) : null;
                // Extract the result specific to the review id
                $extractedResult = $aiReviewSummary ? $aiReviewSummary->get($data[IAiReviewSummary::REVIEW_ID]) : '';
                // Check if the AI call is needed or not for the given user and for give diff
                $aiCallCheckResult = $extractedResult ? $this->checkIfAICallNeeded(
                    $extractedResult,
                    $data,
                    $aiPackage
                ) : null;
                $currentUser       = $this->services->get(ConnectionFactory::USER)->getId();
                if ($aiCallCheckResult && $this->isUserBlocked($aiCallCheckResult, $currentUser)) {
                    $msgRedundantCallSameDiff = sprintf(
                        'Failed to load result as user [%s] has already generated AI data for the diff' .
                        ' of this review %s',
                        $currentUser,
                        $data[IAiAnalysis::REVIEW_ID]
                    );
                    return $this->handleError(
                        $msgRedundantCallSameDiff,
                        Response::STATUS_CODE_500,
                        $msgRedundantCallSameDiff
                    );
                }

                if (!$aiCallCheckResult || (
                    isset($aiCallCheckResult[IAiReviewSummary::CALL_AI_SERVICE])
                    && $aiCallCheckResult[IAiReviewSummary::CALL_AI_SERVICE])) {
                    $processingData = $this->prepareProcessingData($data, $aiPackage);
                    $result         = $aiVendorService->executeAIRequest($processingData);
                    if ($result[self::ERROR]) {
                        return $this->handleError(
                            $result[self::ERROR],
                            $result[self::CODE],
                            $result[self::ERROR]
                        );
                    }
                    $dataReceived    = $result[IAiAnalysisHelper::CONTENT_GENERATED] ?? '';
                    $aiReviewSummary = null;
                    $logger->info(
                        sprintf(
                            "The new AI summary will generated for the User '%s' for the diff of review %s",
                            $currentUser,
                            $data[IAiAnalysis::REVIEW_ID]
                        )
                    );
                } else {
                    // merge the current user into an existing record if AI summary is already generated by other user
                    // for same diff of same version of same review
                    $extractedResult[$data[IAiReviewSummary::FROM_VERSION] . '_' . $data[IAiReviewSummary::TO_VERSION]]
                    [$data[IAiReviewSummary::FILE_ID]]
                    [$data[IAiReviewSummary::DIFF_START] . '_' . $data[IAiReviewSummary::DIFF_END]]
                    [self::DEFAULT_PACKAGE_ID][IAiReviewSummary::USERS] =
                        array_merge($aiCallCheckResult[IAiReviewSummary::USERS], [$currentUser]);
                    $aiReviewSummary->set($data[IAiReviewSummary::REVIEW_ID], $extractedResult);
                    $logger->info(
                        sprintf(
                            "User '%s' will added to the AI summary already generated by other user for same diff 
                            of the review %s",
                            $currentUser,
                            $data[IAiAnalysis::REVIEW_ID]
                        )
                    );
                }

                $aiReviewSummary = $this->saveOrUpdateAnalysis(
                    $aiDAO,
                    $data,
                    $aiPackage,
                    $dataReceived ?? null,
                    $aiReviewSummary ?? null,
                    $result[self::ERROR] ?? null
                );
            }
        } catch (FilterException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_400);
            return $this->error($e->getMessages(), $this->getResponse()->getStatusCode());
        } catch (ErrorException |
            ForbiddenException |
            RecordNotFoundException |
            UnauthorizedException |
            InvalidArgumentException |
            Exception $e
        ) {
            return $this->handleError($e->getMessage(), $this->mapExceptionToStatusCode($e), $e->getMessage());
        }

        return $this->success(
            [self::DATA_AI_SUMMARY => $aiReviewSummary ?
            [$data[IAiReviewSummary::REVIEW_ID] => $aiReviewSummary->get($data[IAiReviewSummary::REVIEW_ID]),
                IAiReviewSummary::ID => $aiReviewSummary->getId()]
            : []]
        );
    }

    /**
     * Discard AI Summary for Swarm
     *
     * This function discards the already generate summary for the logged in swarm user
     * we are performing the soft discard meaning we are just removing the logged-in user entry
     * from the user array
     * @return  JsonModel
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    public function discardAction() : JsonModel
    {
        $aiReviewSummary = new stdClass();
        $logger          = $this->services->get(SwarmLogger::SERVICE);
        try {
            $this->checkUserAuthentication();
            $data     = $this->getData();
            $recordId = $data[IAiReviewSummary::ID];
            $filter   = $this->services->get(IAiAnalysis::DISCARD_ANALYSIS_FILTER);
            $filter->setData($data);

            if (!$filter->isValid()) {
                $logger->trace(
                    "AI:discardAction: Filter Message - " . json_encode($filter->getMessages(), true)
                );
                return $this->handleValidationErrors($filter);
            }
            /** @var AiAnalysisDAO $aiDAO */
            $aiDAO = $this->services->get(IDao::AI_ANALYSIS_DAO);
            $aiDAO->checkRestrictedAccess(
                IAiReviewSummary::REVIEW . '/' . $data[IAiReviewSummary::REVIEW_ID]
            );

            if ($recordId) {
                // Fetch the existing record from db based on id
                $aiReviewSummary = $this->fetchResultIfAvailable($recordId);
                // Extract the result specific to the review id
                $extractedResult = $aiReviewSummary->get($data[IAiReviewSummary::REVIEW_ID]);
                if (is_null($extractedResult)) {
                    throw new NotFoundException(
                        $this->services->get('translator')->t('Cannot fetch entry. Id does not exist')
                    );
                }
                $currentUser = $this->services->get(ConnectionFactory::USER)->getId();
                // Remove the logged-in user from the users array
                $this->removeDiscardedUser($extractedResult, $currentUser);
                $aiReviewSummary->set($data[IAiReviewSummary::REVIEW_ID], $extractedResult);
                $aiReviewSummary = $this->saveOrUpdateAnalysis(
                    $aiDAO,
                    $data,
                    null,
                    null,
                    $aiReviewSummary,
                    null
                );
                $logger->info(
                    sprintf(
                        "AI summary id '%s' of user '%s' for review '%s' is discarded",
                        $recordId,
                        $currentUser,
                        $data[IAiReviewSummary::REVIEW_ID]
                    )
                );
            }
        } catch (ErrorException | ForbiddenException | RecordNotFoundException | UnauthorizedException | Exception $e) {
            return $this->handleError($e->getMessage(), $this->mapExceptionToStatusCode($e), $e->getMessage());
        }
        return $this->success(
            [self::DATA_AI_SUMMARY => $aiReviewSummary ?
                [$data[IAiReviewSummary::REVIEW_ID] => $aiReviewSummary->get($data[IAiReviewSummary::REVIEW_ID])]
                : [], IAiReviewSummary::ID => $aiReviewSummary->getId()]
        );
    }

    /**
     * This API will be used to load package details inside redux store at review initialization
     * @return JsonModel
     */
    public function getAiPackageDetailsAction() : JsonModel
    {
        try {
             $this->checkUserAuthentication();
             $config        = $this->services->get(IConfigDefinition::CONFIG);
             $aiVendors     = ConfigManager::getValue($config, IConfigDefinition::AI_REVIEW_AI_VENDORS);
             $aiPackageData = [];
             // filter the data remove the api_key
            foreach ($aiVendors as $aiVendor) {
                 $aiPackageData[$aiVendor[IConfigDefinition::AI_PACKAGE_ID]] = [
                     IConfigDefinition::AI_PACKAGE_ID => $aiVendor[IConfigDefinition::AI_PACKAGE_ID],
                     IConfigDefinition::AI_VENDOR => $aiVendor[IConfigDefinition::AI_VENDOR],
                     IConfigDefinition::AI_PACKAGE_KEY => $aiVendor[IConfigDefinition::AI_PACKAGE_KEY],
                     IConfigDefinition::AI_PACKAGE_VALUE => $aiVendor[IConfigDefinition::AI_PACKAGE_VALUE],
                     IConfigDefinition::AI_MODEL => $aiVendor[IConfigDefinition::AI_MODEL],
                     IConfigDefinition::AI_PACKAGE_TYPE => $aiVendor[IConfigDefinition::AI_PACKAGE_TYPE],
                     IConfigDefinition::AI_MIN_CHAR_LIMIT => $aiVendor[IConfigDefinition::AI_MIN_CHAR_LIMIT],
                     IConfigDefinition::AI_MAX_CHAR_LIMIT => $aiVendor[IConfigDefinition::AI_MAX_CHAR_LIMIT],
                 ];
            }
            $result = [
                 self::DATA_AI_PACKAGE_DETAILS  => $aiPackageData ?? [],
                 AbstractKey::FETCH_TOTAL_COUNT => count($aiPackageData),
            ];
            return $this->success($result);
        } catch (ForbiddenException | UnauthorizedException | Exception $e) {
            return $this->handleError($e->getMessage(), $this->mapExceptionToStatusCode($e), $e->getMessage());
        }
    }

    /**
     * This function removes logged-in user from the users array who hits discard option
     * @param array $extractedResult
     * @param string $currentUser
     * @return void
     */
    protected function removeDiscardedUser(array &$extractedResult, string $currentUser) : void
    {
        // Check if 'users' key exists at the current level
        if (isset($extractedResult[IAiReviewSummary::USERS]) && is_array($extractedResult[IAiReviewSummary::USERS])) {
            // Search for the username and remove it if found
            $index = array_search($currentUser, $extractedResult[IAiReviewSummary::USERS]);
            if ($index !== false) {
                unset($extractedResult[IAiReviewSummary::USERS][$index]);
                // Re-index the array after deletion
                $extractedResult[IAiReviewSummary::USERS] = array_values($extractedResult[IAiReviewSummary::USERS]);
            } else {
                throw new \InvalidArgumentException(
                    sprintf(
                        "Discard action is invalid, user [%s] has never generated this AI analysis",
                        $currentUser
                    )
                );
            }
        }

        // Iterate over each item in the array
        foreach ($extractedResult as &$value) {
            // If the value is an array, make a recursive call
            if (is_array($value)) {
                $this->removeDiscardedUser($value, $currentUser);
            }
        }
    }

    /**
     * This function check if the user is authenticated or not for accessing the API and if AI config is enabled
     * @return void
     */
    private function checkUserAuthentication(): void
    {
        $checker = $this->services->get(Services::CONFIG_CHECK);
        $checker->checkAll([IPermissions::AUTHENTICATED_CHECKER, IAiReviewSummary::AI_ANALYSIS_CHECKER]);
    }

    /**
     * This function check if the diff content given to the AI model adheres the character limits or not
     * @param $diffContent
     * @return void
     */
    private function checkInputCharacterLimits($diffContent): void
    {
        $checker = $this->services->get(Services::CONFIG_CHECK);
        $checker->check(
            IAiReviewSummary::AI_ANALYSIS_CHAR_LIMIT_CHECKER + [Checker::OPTIONS => [Checker::VALUE => $diffContent]]
        );
    }

    /**
     * This function returns the ai package from the config or from posted data
     * @param array $data
     * @return array|null
     * @throws ConfigException
     */
    private function getAiPackage(array $data): ?array
    {
        $config    = $this->services->get(IConfigDefinition::CONFIG);
        $aiVendors = ConfigManager::getValue($config, IConfigDefinition::AI_REVIEW_AI_VENDORS);
        return $data[IAiAnalysisHelper::AI_PACKAGE] ?? ($aiVendors ? current($aiVendors) : null);
    }

    /**
     * This function sets the status code and returns the error in json format
     * @param $filter
     * @return JsonModel
     */
    private function handleValidationErrors($filter): JsonModel
    {
        $this->getResponse()->setStatusCode(Response::STATUS_CODE_400);
        return $this->error($filter->getMessages(), Response::STATUS_CODE_400);
    }

    /**
     * This function performs the logging of message as well as the handle the errors based on status code
     * @param string $logMessage
     * @param int $statusCode
     * @param string $userMessage
     * @return JsonModel
     */
    private function handleError(
        string $logMessage,
        int $statusCode,
        string $userMessage
    ): JsonModel {
        if (!$userMessage) {
            $userMessage = $logMessage = IAiAnalysisHelper::UNAUTHORIZED;
        }

        if ($logMessage) {
            $logger = $this->services->get(SwarmLogger::SERVICE);
            $logger->debug($logMessage);
        }
        $this->getResponse()->setStatusCode($statusCode);
        return $this->error([$this->buildMessage($statusCode, $userMessage)], $statusCode);
    }

    /**
     * This function will block the user making the redundant call to AI service for the same diff of same version
     * @param array $aiCallCheckResult
     * @param string $currentUser
     * @return bool
     */
    private function isUserBlocked(array $aiCallCheckResult, string $currentUser): bool
    {
        return !$aiCallCheckResult[IAiReviewSummary::CALL_AI_SERVICE]
            && in_array($currentUser, $aiCallCheckResult[IAiReviewSummary::USERS] ?? []);
    }

    /**
     * This function will prepare the data for the AI service and the data to be saved in swarm ai summary
     * @param array $data
     * @param array $aiPackage
     * @return array
     */
    private function prepareProcessingData(array $data, array $aiPackage): array
    {
        return [
            IConfigDefinition::AI_MODEL => $aiPackage[IConfigDefinition::AI_MODEL],
            IConfigDefinition::AI_PACKAGE_TYPE => $aiPackage[IConfigDefinition::AI_PACKAGE_TYPE],
            IAiAnalysisHelper::CONTENT_TO_ANALYZE => $data[AbstractAiAdapter::DATA_DIFF_CONTENT],
            IAiAnalysis::FILE_NAME => $data[IAiAnalysis::FILE_NAME]
        ];
    }

    /**
     * This function fetches the record for the given
     * @param int $id
     * @return AiReviewSummary
     * @throws ForbiddenException
     */
    private function fetchResultIfAvailable(int $id): AiReviewSummary
    {
        /** @var AiAnalysisDAO $aiDAO */
        $aiDAO = $this->services->get(IDao::AI_ANALYSIS_DAO);
        return $aiDAO->fetchIfResultAvailable($id);
    }

    /**
     * This is wrapper function to either update the existing the record of ai review summary or create the new
     * record for the provided data
     * @param AiAnalysisDAO $aiDAO
     * @param array $data
     * @param array $aiPackage
     * @param string|null $dataReceived
     * @param AiReviewSummary|null $aiReviewSummary
     * @param string|null $error
     * @return AiReviewSummary
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    private function saveOrUpdateAnalysis(
        AiAnalysisDAO $aiDAO,
        array $data,
        ?array $aiPackage,
        ?string $dataReceived,
        ?AiReviewSummary $aiReviewSummary,
        ?string $error
    ): AiReviewSummary {
        return $aiDAO->saveOrUpdate($data, $aiPackage, $dataReceived, $aiReviewSummary, $error);
    }

    /**
     * This function maps exception to the status code
     * @param Exception $e
     * @return int
     */
    private function mapExceptionToStatusCode(Exception $e): int
    {
        if ($e instanceof ForbiddenException) {
            return Response::STATUS_CODE_403;
        } elseif ($e instanceof UnauthorizedException) {
            return Response::STATUS_CODE_401;
        } elseif ($e instanceof InvalidArgumentException) {
            return Response::STATUS_CODE_400;
        } elseif ($e instanceof RecordNotFoundException) {
            return Response::STATUS_CODE_404;
        }
        return Response::STATUS_CODE_500;
    }


    /**
     * Get the posted data and put into a format to be used.
     * @return mixed
     */
    protected function getData()
    {
        // This isn't setting session data correctly so login not remembered when going to server context
        $request = $this->getRequest();
        if ($this->requestHasContentType($request, self::CONTENT_TYPE_JSON)) {
            $requestData = json_decode($request->getContent(), true);
        } else {
            $requestData = $request->getPost()->toArray();
        }
        // Get the baseurl and scheme for the authHelper, so we do not have to pass request in.
        $requestData[IAiReviewSummary::BASE_URL] = $request->getBaseUrl();
        $requestData[IAiReviewSummary::HTTPS]    = $request instanceof Request
            && $request->getUri()->getScheme() == IAiReviewSummary::HTTPS;
        return $requestData;
    }

    /**
     * Checks whether call to AI vendor is needed or not
     *  This will check for given reviewId, toVersion, fromVersion, fileId, difStart, diffEnd,
     *  ai_package_id if it matches then it returns the array with list of users to update the record else blank array
     * @param array $resultAvailable
     * @param array $postedData
     * @param array $aiPackage
     * @return array
     */
    protected function checkIfAICallNeeded(array $resultAvailable, array $postedData, array $aiPackage): array
    {
        $fromVersion = $postedData[IAiReviewSummary::FROM_VERSION];
        $toVersion   = $postedData[IAiReviewSummary::TO_VERSION];
        $fileId      = $postedData[IAiReviewSummary::FILE_ID];
        $diffStart   = $postedData[IAiReviewSummary::DIFF_START];
        $diffEnd     = $postedData[IAiReviewSummary::DIFF_END];
        $versionsKey = $fromVersion . '_' . $toVersion;
        if (!isset($resultAvailable[$versionsKey])) {
            return $this->defaultResponse(IAiReviewSummary::REVIEW_ID);
        }

        if (!isset($resultAvailable[$versionsKey][$fileId])) {
            return $this->defaultResponse(IAiReviewSummary::FROM_VERSION);
        }

        $diffKey = $diffStart . '_' . $diffEnd;
        if (!isset($resultAvailable[$versionsKey][$fileId][$diffKey])) {
            return $this->defaultResponse(IAiReviewSummary::FILE_ID);
        }

        if (!isset($resultAvailable[$versionsKey][$fileId][$diffKey][self::DEFAULT_PACKAGE_ID])) {
            return $this->defaultResponse(IAiReviewSummary::DIFF_START);
        }

        $resultPackageType = '';
        $resultAIModel     = '';
        if (isset(
            $resultAvailable[$versionsKey][$fileId][$diffKey][self::DEFAULT_PACKAGE_ID]
            [IAiReviewSummary::AI_PACKAGE_TYPE]
        )) {
            $resultPackageType = $resultAvailable[$versionsKey][$fileId][$diffKey]
            [self::DEFAULT_PACKAGE_ID][IAiReviewSummary::AI_PACKAGE_TYPE];
        }

        if (isset(
            $resultAvailable[$versionsKey][$fileId][$diffKey][self::DEFAULT_PACKAGE_ID]
            [IAiReviewSummary::AI_MODEL]
        )) {
            $resultAIModel = $resultAvailable[$versionsKey][$fileId][$diffKey]
            [self::DEFAULT_PACKAGE_ID][IAiReviewSummary::AI_MODEL];
        }


        if (!$resultPackageType || $resultPackageType !== $aiPackage[IConfigDefinition::AI_PACKAGE_TYPE]) {
            return $this->defaultResponse(IAiReviewSummary::AI_PACKAGE_TYPE);
        }

        if ($aiPackage[IConfigDefinition::AI_VENDOR] === Services::OPEN_AI && ( !$resultAIModel ||
                $resultAIModel !== $aiPackage[IConfigDefinition::AI_MODEL])) {
            return $this->defaultResponse(IAiReviewSummary::AI_MODEL);
        }

        $matchedData = $resultAvailable[$versionsKey][$fileId][$diffKey][self::DEFAULT_PACKAGE_ID];

        return [
            IAiReviewSummary::CALL_AI_SERVICE => false,
            IAiReviewSummary::UPDATE_RECORD => true,
            IAiReviewSummary::USERS => $matchedData[IAiReviewSummary::USERS],
            IAiReviewSummary::CONTENT => $matchedData[IAiReviewSummary::CONTENT],
            IAiReviewSummary::RECORD_MATCH_AT => self::DEFAULT_PACKAGE_ID,
        ];
    }

    /**
     * Returns the default response for function checkIfAICallNeeded
     * @param string $recordMatchedAt
     * @return array
     */
    private function defaultResponse(string $recordMatchedAt = ''): array
    {
        return [
            IAiReviewSummary::CALL_AI_SERVICE => true,
            IAiReviewSummary::RECORD_MATCH_AT => $recordMatchedAt,
            IAiReviewSummary::UPDATE_RECORD => true,
            IAiReviewSummary::USERS => [],
        ];
    }
}
