<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.1/2745343
 */
namespace Reviews;

use Application\Config\IDao;
use Application\Connection\ConnectionFactory;
use Laminas\ServiceManager\ServiceManager;
use P4\Log\Logger;
use Projects\Validator\ID;
use Redis\Model\GroupDAO;
use Redis\Model\UserDAO;
use Reviews\Model\IReview;

/**
 * Trait ReviewTrait. Useful functions for dealing with a review
 * @package Reviews
 */
trait ReviewTrait
{
    /**
     * Work out if the change in test status is notable. A notable change is a transition between overall pass and
     * overall fail and vice-versa
     * @param mixed     $status         current test status
     * @param mixed     $newStatus      new test status
     * @return bool if the change of status is considered notable
     */
    public static function isNotableTestStatusChange($status, $newStatus) : bool
    {
        Logger::log(
            Logger::TRACE,
            sprintf("%s: Status is [%s], newStatus is [%s]", ReviewTrait::class, $status, $newStatus)
        );
        $notable = $newStatus ?? false;
        if ($notable) {
            $notable = ($newStatus === IReview::TEST_STATUS_FAIL && $status !== IReview::TEST_STATUS_FAIL) ||
                       ($newStatus === IReview::TEST_STATUS_PASS && $status === IReview::TEST_STATUS_FAIL);
        }
        Logger::log(
            Logger::TRACE,
            sprintf("%s: Value of notable test change is [%s]", ReviewTrait::class, $notable)
        );
        return $notable;
    }

    /**
     * Remove the deleted participants (users/groups) from the review array
     * @param string $key
     * @param array $participants
     * @param null $services
     * @return array
     */
    public function filterDeletedParticipants(string $identifier, array $participants, $services = null): array
    {
        $groupPrefix = 'swarm-group-';
        if ($identifier === IReview::FIELD_PARTICIPANTS_DATA) {
            $participantsCopy = $participants;
            $participants     = array_keys($participants);
        }
        /** @var UserDAO $userDao */
        $userDao = $services->get(IDao::USER_DAO);
        /** @var GroupDAO $groupDao */
        $groupDao = $services->get(IDao::GROUP_DAO);
        $p4       = $services->get(ConnectionFactory::P4_ADMIN);
        foreach ($participants as $key => $participant) {
            if (!$userDao->exists($participant, $p4) &&
                !$groupDao->exists(str_replace($groupPrefix, '',  $participant), $p4)) {
                        unset($participants[$key]);
            }
        }

        if ($identifier === IReview::FIELD_PARTICIPANTS_DATA) {
            $participants = array_intersect_key($participantsCopy, array_flip($participants));
        }

        return $participants;
    }
}
