<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.1/2745343
 */
namespace Reviews\Listener;

use Application\Config\IConfigDefinition;
use Application\Log\SwarmLogger;
use Events\Listener\AbstractEventListener;
use Laminas\EventManager\Event;
use Application\Config\ConfigManager;
use Application\Config\ConfigException;
use Reviews\Model\IReview;
use Reviews\Service\IStatistics;

/**
 * Class Statistics. A listener to calculate review statistics
 * @package Reviews\Listener
 */
class Statistics extends AbstractEventListener
{
    /**
     * Calculate a complexity value on review content change
     * @param Event $event  the event
     */
    public function reviewChanged(Event $event)
    {
        $class      = get_class($this);
        $logger     = $this->services->get(SwarmLogger::SERVICE);
        $data       = $event->getParam('data');
        $dataString = is_array($data) ? json_encode($data) : (string)$data;
        $logger->debug(sprintf("%s: starting to check complexitiy", $class, $dataString));
        // task.review is fired for state change etc., we only want to recalculate if an update has occurred
        // or a file or files have been deleted from the change list
        if ($data && (isset($data['updateFromChange']) || isset($data[IReview::DELFROMCHANGE]))) {
            // We want to deal with the review that the review listener has been dealing with as we are planning to
            // make changes and save
            $review = $event->getParam('review');
            $logger->debug(sprintf("%s: data is, %s", $class, json_encode($data, true)));
            $logger->debug(sprintf("%s: review is, %s", $class, json_encode($review->toArray(), true)));
            // On some occasions due to the order of events this listener may be called twice for the same review. It
            // can be called in several ways - via API, using #review or updating a review. We need to check
            // a number of conditions here, to prevent statistics being called twice when an API is used (via P4V)
            // whilst still allowing #review creation and updates to operate correctly
            $shouldCalculate = !isset($data['isAdd'])
                || $data['isAdd'] === true
                || ($data['isAdd'] === false && $review->getHeadVersion() > 0);
            if ($shouldCalculate) {
                $logger->trace(sprintf("%s: Calculating complexity for review [%s]", $class, $review->getId()));
                $complexityService = $this->services->get(IStatistics::COMPLEXITY_SERVICE);
                $complexityService->calculateComplexity($review);
                $logger->trace(sprintf("%s: Calculated complexity for review [%s]", $class, $review->getId()));
            } else {
                $logger->trace(sprintf("%s: Skipping complexity for review [%s]", $class, $review->getId()));
            }
        }
    }

    /**
     * Attaches this event only if the complexity calculation configuration value is set to default to use the
     * Swarm implementation
     * @param mixed $eventName      the event name
     * @param array $eventDetail    the event detail
     * @return bool true if the complexity calculation configuration value is 'default', otherwise false
     * @throws ConfigException
     */
    public function shouldAttach($eventName, $eventDetail)
    {
        $config = $this->services->get(IConfigDefinition::CONFIG);
        return ConfigManager::getValue(
            $config,
            IConfigDefinition::REVIEWS_COMPLEXITY_CALCULATION,
            IConfigDefinition::DEFAULT
        ) === IConfigDefinition::DEFAULT;
    }
}
