<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.1/2745343
 */

namespace Authentication\Controller;

use Api\Controller\AbstractRestfulController;
use Application\Config\ConfigException;
use Application\Config\ConfigManager;
use Application\Config\IConfigDefinition;
use Application\Config\IDao;
use Application\Connection\ConnectionFactory;
use Application\Model\IModelDAO;
use Application\Permissions\Exception\BasicAuthFailedException;
use Application\Permissions\Exception\UnauthorizedException;
use Authentication\Factory\LoginServiceFactory;
use Authentication\Helper\ILoginHelper;
use Application\Log\SwarmLogger;
use Application\Filter\FormBoolean;
use Application\I18n\TranslatorFactory;
use Authentication\Helper\ILogoutHelper;
use Authentication\Helper\LoginHelper;
use Authentication\Helper\LogoutHelper;
use Authentication\Service\SsoLoginAdapter;
use Laminas\Http\Request;
use Laminas\Http\Response;
use Laminas\ServiceManager\ServiceManager;
use Laminas\Stdlib\Parameters;
use Laminas\View\Model\JsonModel;
use P4\Connection\Exception\ServiceNotFoundException;
use P4\Exception;
use Redis\Model\UserDAO;
use Users\Model\IUser;
use Users\Model\User;

/**
 * Class AuthenticationApi
 * @package Authentication\Controller
 */
class AuthenticationApi extends AbstractRestfulController
{
    const LOG_PREFIX    = AuthenticationApi::class;
    const METHOD_PREFIX = 'auth_';

    protected $services;

    /**
     * @param ServiceManager $services
     */
    public function __construct(ServiceManager $services)
    {
        $this->services = $services;
    }

    /**
     * Login to Swarm using the login_service_type provided
     * & redirecting to appropriate login system & returning the respective response
     */
    public function loginAction(): JsonModel
    {
        $data           = $this->getData();
        $isSamlResponse = isset($_REQUEST[ILoginHelper::SAML_RESPONSE]);
        if ($isSamlResponse) {
            return $this->handleSamlResponse($data);
        }
        $logger = $this->services->get(SwarmLogger::SERVICE);
        // Get the requested login service type from the POST request body.
        $method       = $data[ILoginHelper::METHOD] ?? null;
        $method       = $method ? self::METHOD_PREFIX . $method : LoginServiceFactory::BASICLOGINADAPTER;
        $loginService = $this->services->get($method);

        $logger->info(
            sprintf(
                "[%s]: For '%s' user requested login service type is '%s'",
                self::LOG_PREFIX,
                $data[ILoginHelper::USERNAME],
                $method
            )
        );

        //Call the logic for invoking the requested login service type inside the LoginServiceFactory
        /** @var JsonModel $response */
        $response   = $loginService->login($data);
        $statusCode = $response->getVariable('error') ?? Response::STATUS_CODE_200;
        if ($statusCode) {
            $this->getResponse()->setStatusCode($statusCode);
        }
        return $response;
    }

    /**
     * Function to handle the saml response returned since we need to keep it
     * align with the older api hence we have added it in side this API
     * @param $data array request data
     * @return JsonModel
     */
    private function handleSamlResponse(array $data): JsonModel
    {
        $msg        = [];
        $translator = $this->services->get(TranslatorFactory::SERVICE);
        try {
            $this->loginResponseAction($data);
            $msg[] = self::buildMessage(
                ILoginHelper::CODE_LOGIN_SUCCESSFUL,
                $translator->t(ILoginHelper::TEXT_LOGIN_SUCCESSFUL)
            );
            $user  = $this->currentUser();
            return self::success([IUser::USER => [$user]], $msg);
        } catch (BasicAuthFailedException $error) {
            $msg[] = self::buildMessage(ILoginHelper::CODE_LOGIN_INCORRECT, $translator->t($error->getMessage()));
        } catch (UnauthorizedException $error) {
            $msg[] = self::buildMessage(ILoginHelper::CODE_NOT_LOGGED_IN, $translator->t($error->getMessage()));
        } catch (ServiceNotFoundException $error) {
            $msg[] = self::buildMessage(ILoginHelper::CODE_SWARM_CONNECTION, $translator->t($error->getMessage()));
        } catch (\P4\Exception $error) {
            $msg[] = self::buildMessage(ILoginHelper::CODE_LOGIN_MISSING, $translator->t($error->getMessage()));
        } catch (\Exception $error) {
            $msg[] = self::buildMessage(ILoginHelper::USER_ERROR, $translator->t($error->getMessage()));
        }
        $this->getResponse()->setStatusCode($error->getCode());
        return self::error($msg, $this->getResponse()->getStatusCode());
    }
    /**
     * Login response to swarm when saml sent the data back to swarm via HAS response
     * @param $data array request data
     * @throws ConfigException
     * @throws ServiceNotFoundException
     */
    public function loginResponseAction($data)
    {
        $session = $this->services->get(ILoginHelper::SESSION);
        /** @var LoginHelper $helper */
        $helper = $this->services->get(ILoginHelper::HELPER_NAME);
        // Get the requested login service type from the POST request body.
        $loginService = isset($data[ILoginHelper::SAML_RESPONSE])
            ? $this->services->get(LoginServiceFactory::SAMLLOGINADAPTER) :
            $this->services->get($data[ILoginHelper::METHOD]);
        // Clear any existing session data on login, we need to explicitly restart the session
        $session->start();
        $session->getStorage()->clear();
        $p4Admin       = $this->services->get(ConnectionFactory::P4_ADMIN);
        $candidateAuth = $loginService->handleSamlResponse();
        $authUser      = $candidateAuth[ILoginHelper::AUTH_USER];

        if ($candidateAuth[ILoginHelper::IS_VALID]) {
            $helper->setCookieInformation($data);
            $helper->invalidateCache($authUser, $p4Admin);
        } else {
            try {
                /* verify that the admin connection is still valid;
                if so then the user credentials provided are incorrect.*/
                $userDao = $this->services->get(IDao::USER_DAO);
                $userDao->fetchById($p4Admin->getUser(), $p4Admin);
                throw new BasicAuthFailedException(ILoginHelper::TEXT_LOGIN_INCORRECT, Response::STATUS_CODE_401);
            } catch (\Exception $exception) {
                if ($exception instanceof BasicAuthFailedException) {
                    throw $exception;
                }
                throw new ServiceNotFoundException(ILoginHelper::TEXT_SWARM_CONNECTION, Response::STATUS_CODE_500);
            }
        }
        $session->writeClose();
        // Redirect to the page that the login was initiated from, or if that is not set
        // redirect to the route defined in routes as 'home'
        $url = $helper->getLoginReferrer();
        if (!$url) {
            $url = $this->services->get('ViewHelperManager')->get('url')->__invoke('home');
        }
        $this->getResponse()->getHeaders()->addHeaderLine('Location', $url);
        $this->getResponse()->setStatusCode(302);
        return true;
    }

    /**
     * Logout of Swarm
     *
     * @return JsonModel
     * @throws ConfigException
     */
    public function logoutAction(): JsonModel
    {
        $request = $this->getRequest();
        /* If we want to be redirected do it. */
        $filter = new FormBoolean([FormBoolean::NULL_AS_FALSE => false]);
        /** @var LogoutHelper $helper */
        $helper   = $this->services->get(ILogoutHelper::HELPER_NAME);
        $redirect =  $filter->filter($request->getQuery(ILogoutHelper::PARAM_REDIRECT));
        $request->setQuery(new Parameters([ILogoutHelper::PARAM_REDIRECT => $redirect]));
        try {
            $data = $helper->logout($request, $this->getResponse());
        } catch (Exception $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_500);
            return self::error(
                [self::buildMessage(ILogoutHelper::CODE_LOGOUT_ERROR, $e->getMessage())],
                $this->getResponse()->getStatusCode()
            );
        }
        return self::success(
            $data,
            [
                self::buildMessage(
                    ILogoutHelper::CODE_LOGOUT_SUCCESSFUL,
                    $this->services->get(TranslatorFactory::SERVICE)->t(ILogoutHelper::TEXT_LOGOUT_SUCCESSFUL)
                )
            ]
        );
    }

    /**
     * Get the data provided and put into a format to be used.
     * @return mixed
     */
    protected function getData()
    {
        // This isn't setting session data correctly so login not remembered when going to server context
        $request = $this->getRequest();
        if ($this->requestHasContentType($request, self::CONTENT_TYPE_JSON)) {
            $requestData = json_decode($request->getContent(), true);
        } else {
            $requestData = $request->getPost()->toArray();
        }
        // Get the baseurl and scheme for the authHelper, so we do not have to pass request in.
        $requestData[ILoginHelper::BASE_URL] = $request->getBaseUrl();
        $requestData[ILoginHelper::HTTPS]    = $request instanceof Request
            && $request->getUri()->getScheme() == 'https';
        return $requestData;
    }

    /**
     * Get the authentication details relating to the current session
     *
     * @return JsonModel
     * @throws ConfigException
     */
    public function getAuthenticationDetailsAction(): JsonModel
    {
        $services   = $this->services;
        $translator = $services->get(TranslatorFactory::SERVICE);
        try {
            $user = $this->currentUser();
            return self::success([IUser::USER => [$user]]);
        } catch (UnauthorizedException $error) {
            $config       = $services->get(IConfigDefinition::CONFIG);
            $requireLogin = ConfigManager::getValue(
                $config,
                IConfigDefinition::SECURITY_REQUIRE_LOGIN,
                true
            );
            if ($requireLogin) {
                $msg[] = self::buildMessage(
                    ILoginHelper::CODE_REQUIRE_LOGIN,
                    $translator->t(ILoginHelper::TEXT_REQUIRE_LOGIN)
                );
            } else {
                $msg[] = self::buildMessage(ILoginHelper::CODE_NOT_LOGGED_IN, $translator->t($error->getMessage()));
            }
        } catch (Exception $error) {
            $msg[] = self::buildMessage(ILoginHelper::USER_ERROR, $translator->t($error->getMessage()));
        }
        $this->getResponse()->setStatusCode($error->getCode());
        /** @var LoginHelper $helper */
        $helper = $services->get(ILoginHelper::HELPER_NAME);
        $sso    = $helper->getSSO(P4_SERVER_ID);
        return self::error($msg, $this->getResponse()->getStatusCode(), ['sso' => $sso, 'saml' => $helper->getSaml()]);
    }

    /**
     * Check if the user is logged in and then return the user object
     *
     * @throws UnauthorizedException
     * @throws Exception
     * @throws \Exception
     */
    protected function currentUser()
    {
        /** @var UserDAO $userDao */
        $userDao = $this->services->get(IModelDAO::USER_DAO);
        try {
            $p4User = $this->services->get(ConnectionFactory::P4_USER);
            if ($p4User->isAuthenticated()) {
                $authUser = $userDao->fetchAuthUser($p4User->getUser(), $p4User);
            } else {
                throw new UnauthorizedException(ILoginHelper::TEXT_NOT_LOGGED_IN, Response::STATUS_CODE_401);
            }
        } catch (\Exception $e) {
            if (strpos($e->getMessage(), ILoginHelper::SERVICE_P4_USER_NOT_CREATED) === 0
                || $e instanceof UnauthorizedException) {
                throw new UnauthorizedException(ILoginHelper::TEXT_NOT_LOGGED_IN, Response::STATUS_CODE_401);
            } else {
                throw new \Exception($e->getMessage(), Response::STATUS_CODE_500);
            }
        }

        return $authUser;
    }

    /**
     * Retrieve the url where the user have to be navigated for sso login from redis cache.
     * @return JsonModel
     */
    public function getSsoUrlAction(): JsonModel
    {
        $postData = $this->getData();
        $logger   = $this->services->get(SwarmLogger::SERVICE);
        if (!$postData['username']) {
            $logger->info(
                sprintf(
                    "Failed to get the SSO navigation URL as username posted is invalid",
                    self::LOG_PREFIX
                )
            );
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_400);
            $errors = [
                $this->buildMessage(Response::STATUS_CODE_400, 'username is required and cant be empty')
            ];
            return $this->error($errors, $this->getResponse()->getStatusCode());
        }
        $username = $postData['username'];
        /** @var SsoLoginAdapter $loginService */
        $loginService = $this->services->get(LoginServiceFactory::SSOLOGINADAPTER);
        //Call the logic for invoking the requested login service type inside the LoginServiceFactory
        /** @var JsonModel $response */
        $response   = $loginService->getSsoUrl($username);
        $statusCode = $response->getVariable('error') ?? Response::STATUS_CODE_200;
        $this->getResponse()->setStatusCode($statusCode);
        return $response;
    }
}
