<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.1/2745343
 */

namespace Application\Helper;

use Application\Log\SwarmLogger;
use Application\Model\ServicesModelTrait;
use Files\MimeType;
use kornrunner\Blurhash\Blurhash as ImageBlurhash;
use Throwable;

/**
 * The SwarmBlurHash to be a wrapper of all blur creations.
 * Currently, only supports image blur creations.
 */
class SwarmBlurHash
{
    const LOG_PREFIX = SwarmBlurHash::class;
    use ServicesModelTrait;

    /**
     * Log an exception as a warning
     * @param Throwable $exception
     * @return void
     */
    private static function logException(Throwable $exception)
    {
        $services = ServicesModelTrait::getServices();
        $logger   = $services->get(SwarmLogger::SERVICE);
        $logger->warn(
            sprintf(
                "[%s]: CreateBlur failed due to: %s",
                self::LOG_PREFIX,
                $exception->getMessage()
            )
        );
    }

    /**
     * Create a blur for the file being uploaded.
     * @param string    $fileType           the file type
     * @param string    $fileLocation       the file location
     * @param int       $width              width for the image scale, default to 240
     * @return string the blur if created - or null if not
     */
    public static function createBlur(string $fileType, string $fileLocation, int $width = 240): ?string
    {
        try {
            return (MimeType::isSimpleImageType($fileType) ? static::createImageBlur($fileLocation, $width) : null);
        } catch (Throwable $exception) {
            self::logException($exception);
        }
        return null;
    }

    /**
     * Create the image blur from the file that we have been provided.
     *
     * @param string    $imageFilePath      The path to the file on swarm disk.
     * @param int       $width              width for the image scale
     * @return string the blur hash string.
     */
    protected static function createImageBlur(string $imageFilePath, int $width): string
    {
        return self::createImageBlurFromString(file_get_contents($imageFilePath), $width);
    }

    /**
     * Create a blur from a string representation of an image
     * @param string    $imageString    the image as a string
     * @param int       $width          width for the image scale
     * @return string
     */
    public static function createImageBlurFromString(string $imageString, int $width = 240): string
    {
        $image = imagecreatefromstring($imageString);
        $image = imagescale($image, $width);
        return self::createImageBlurFromImage($image);
    }

    /**
     * Create the image blur from a scaled image.
     * @param mixed $image The scaled image.
     * @return string the blur hash string.
     */
    public static function createImageBlurFromImage($image): ?string
    {
        try {
            $width  = imagesx($image);
            $height = imagesy($image);
            $pixels = [];
            for ($y = 0; $y < $height; ++$y) {
                $row = [];
                for ($x = 0; $x < $width; ++$x) {
                    $index  = imagecolorat($image, $x, $y);
                    $colors = imagecolorsforindex($image, $index);

                    $row[] = [$colors['red'], $colors['green'], $colors['blue']];
                }
                $pixels[] = $row;
            }

            $components_x = 4;
            $components_y = 3;
            return ImageBlurhash::encode($pixels, $components_x, $components_y);
        } catch (Throwable $exception) {
            self::logException($exception);
        }
        return null;
    }
}
